"""
Interface graphique du synthétiseur MIDI avec CustomTkinter.
"""

import customtkinter as ctk
import mido
import threading
import os
from synth_engine import SynthEngine
from sf2_engine import FluidSynthEngine
from audio_output import AudioOutput
from presets import PresetManager
from config_manager import ConfigManager
from preset_editor import PresetEditorWindow
from sequencer_engine import SequencerEngine
from sequencer_gui import SequencerGUI

# Configuration de l'apparence CustomTkinter
ctk.set_appearance_mode("dark")
ctk.set_default_color_theme("blue")

class SynthGUI(ctk.CTk):
    """
    Interface graphique principale du synthétiseur.
    """
    
    def __init__(self):
        super().__init__()
        
        # Configuration de la fenêtre
        self.title("🎹 Synthétiseur MIDI Python")
        self.geometry("1200x800")  # Légèrement plus grand pour le séquenceur
        
        # Créer le menu avant tout
        self._create_menu()
        
        # Gestionnaires
        self.config_manager = ConfigManager()
        self.preset_manager = PresetManager()
        self.synth_engine = SynthEngine(sample_rate=44100, num_voices=32)
        self.sf2_engine = FluidSynthEngine(sample_rate=44100)
        self.current_engine = self.synth_engine
        self.last_applied_synth_preset = (None, None) # (cat, name)
        
        # Séquenceur
        self.sequencer_engine = SequencerEngine(bpm=120, ppqn=96, sample_rate=44100)
        self.sequencer_gui = None  # Sera créé dans _create_widgets
        
        # Initialiser la sortie audio avec les deux moteurs (Mixage permanent)
        self.audio_output = AudioOutput(
            self.synth_engine, 
            sf2_engine=self.sf2_engine,
            sample_rate=44100
        )
        self.audio_output.start()
        
        self.midi_port = None
        self.midi_thread = None
        self.is_running = False
        
        # Thread pour mise à jour du status
        self.status_thread = None
        self.status_running = False
        
        # Appliquer la configuration sauvegardée
        self._apply_saved_config()
        
        # Créer l'interface
        self._create_widgets()
        
        # Charger le preset par défaut
        last_category = self.config_manager.get_last_category()
        last_preset = self.config_manager.get_last_preset()
        self._load_preset(last_category, last_preset)
        
        # Connexion automatique au port MIDI sauvegardé
        self._auto_connect_midi()
        
        # Charger la SoundFont par défaut si elle existe
        self._load_default_sf2()
        
        # Restaurer la configuration SoundFont sauvegardée (moteur, SoundFont, instrument)
        self._restore_soundfont_config()
        
        # Démarrer la mise à jour du status
        self._start_status_updates()
        
        # Gérer la fermeture
        self.protocol("WM_DELETE_WINDOW", self._on_closing)
    
    def _apply_saved_config(self):
        """Applique la configuration sauvegardée."""
        # Position et taille de la fenêtre
        pos = self.config_manager.get_window_position()
        size = self.config_manager.get_window_size()
        self.geometry(f"{size['width']}x{size['height']}+{pos['x']}+{pos['y']}")

    def _create_menu(self):
        """Crée la barre de menu."""
        from tkinter import Menu, filedialog,messagebox
        
        menubar = Menu(self)
        self.config(menu=menubar)
        
        # Menu Projet (pour le séquenceur)
        project_menu = Menu(menubar, tearoff=0)
        menubar.add_cascade(label="Projet", menu=project_menu)
        
        project_menu.add_command(label="Nouveau projet", command=self._on_new_project)
        project_menu.add_command(label="Ouvrir projet...", command=self._on_open_project)
        project_menu.add_command(label="Sauvegarder projet...", command=self._on_save_project)
        project_menu.add_separator()
        project_menu.add_command(label="Exporter MIDI...", command=self._on_export_midi)
    
    def _create_widgets(self):
        """Crée tous les widgets de l'interface."""
        
        # Configuration de la grille - 4 colonnes horizontales
        self.grid_columnconfigure(0, weight=1)  # Oscillateur
        self.grid_columnconfigure(1, weight=1)  # Chorus
        self.grid_columnconfigure(2, weight=1)  # Reverb
        self.grid_columnconfigure(3, weight=1)  # Volume
        self.grid_rowconfigure(0, weight=0)  # Header
        self.grid_rowconfigure(1, weight=0)  # MIDI Config
        self.grid_rowconfigure(2, weight=0)  # Presets
        self.grid_rowconfigure(3, weight=1)  # Controls
        self.grid_rowconfigure(4, weight=0)  # Footer
        
        # ===== HEADER =====
        self.header_frame = ctk.CTkFrame(self, fg_color="transparent")
        self.header_frame.grid(row=0, column=0, columnspan=4, padx=10, pady=(10, 5), sticky="ew")
        
        self.title_label = ctk.CTkLabel(
            self.header_frame,
            text="🎹 Synthétiseur MIDI",
            font=("Arial", 24, "bold")
        )
        self.title_label.pack(side="left", padx=10)
        
        self.connection_indicator = ctk.CTkLabel(
            self.header_frame,
            text="● Déconnecté",
            text_color="red",
            font=("Arial", 14)
        )
        self.connection_indicator.pack(side="right", padx=10)
        
        # ===== MIDI CONFIGURATION =====
        self.midi_frame = ctk.CTkFrame(self)
        self.midi_frame.grid(row=1, column=0, columnspan=4, padx=10, pady=5, sticky="ew")
        
        ctk.CTkLabel(
            self.midi_frame,
            text="Port MIDI:",
            font=("Arial", 12)
        ).grid(row=0, column=0, padx=10, pady=10, sticky="w")
        
        self.midi_port_var = ctk.StringVar()
        self.midi_dropdown = ctk.CTkOptionMenu(
            self.midi_frame,
            variable=self.midi_port_var,
            values=["Aucun port détecté"],
            command=self._on_midi_port_selected,
            width=500
        )
        self.midi_dropdown.grid(row=0, column=1, padx=5, pady=10, sticky="ew", columnspan=2)
        
        # Buffer Size Selector
        ctk.CTkLabel(
            self.midi_frame,
            text="Buffer Audio:",
            font=("Arial", 12)
        ).grid(row=1, column=0, padx=10, pady=5, sticky="w")
        
        self.buffer_size_var = ctk.StringVar(value="512")
        self.buffer_dropdown = ctk.CTkOptionMenu(
            self.midi_frame,
            variable=self.buffer_size_var,
            values=["128", "256", "512", "1024", "2048"],
            command=self._on_buffer_size_changed,
            width=150
        )
        self.buffer_dropdown.grid(row=1, column=1, padx=5, pady=5, sticky="w")
        
        ctk.CTkLabel(
            self.midi_frame,
            text="(Plus petit = moins de latence, plus grand = plus stable)",
            font=("Arial", 9),
            text_color="gray"
        ).grid(row=1, column=2, padx=5, pady=5, sticky="w")
        
        self.midi_frame.grid_columnconfigure(1, weight=1)
        
        # Charger les ports MIDI au démarrage
        self._refresh_midi_ports()
        
        # ===== TABS SYSTEM =====
        self.tabview = ctk.CTkTabview(self, command=self._on_tab_changed)
        self.tabview.grid(row=2, column=0, columnspan=4, padx=10, pady=5, sticky="nsew")
        
        self.tab_synth = self.tabview.add("Synthétiseur")
        self.tab_sf2 = self.tabview.add("SoundBank")
        self.tab_sequencer = self.tabview.add("Séquenceur")
        
        # Configuration des colonnes dans les onglets
        self.tab_synth.grid_columnconfigure(0, weight=1)
        self.tab_sf2.grid_columnconfigure(0, weight=1)
        self.tab_sequencer.grid_columnconfigure(0, weight=1)
        self.tab_sequencer.grid_rowconfigure(0, weight=1)

        # ===== PRESETS (Dans l'onglet Synthétiseur) =====
        self.preset_frame = ctk.CTkFrame(self.tab_synth)
        self.preset_frame.grid(row=0, column=0, padx=10, pady=10, sticky="ew")
        
        # Ligne 1: Catégorie
        ctk.CTkLabel(
            self.preset_frame,
            text="Catégorie:",
            font=("Arial", 12)
        ).grid(row=0, column=0, padx=10, pady=5, sticky="w")
        
        self.category_var = ctk.StringVar(value=self.config_manager.get_last_category())
        categories = self.preset_manager.get_categories()
        self.category_dropdown = ctk.CTkOptionMenu(
            self.preset_frame,
            variable=self.category_var,
            values=categories,
            command=self._on_category_selected,
            width=400
        )
        self.category_dropdown.grid(row=0, column=1, padx=5, pady=5, sticky="ew")
        
        # Ligne 2: Preset
        ctk.CTkLabel(
            self.preset_frame,
            text="Preset:",
            font=("Arial", 12)
        ).grid(row=1, column=0, padx=10, pady=5, sticky="w")
        
        self.preset_var = ctk.StringVar(value=self.config_manager.get_last_preset())
        
        # Créer un scrollable dropdown pour les presets
        initial_category = self.category_var.get()
        preset_names = self.preset_manager.get_presets_in_category(initial_category)
        
        self.preset_dropdown = ctk.CTkOptionMenu(
            self.preset_frame,
            variable=self.preset_var,
            values=preset_names if preset_names else ["Aucun preset"],
            command=self._on_preset_selected,
            width=400
        )
        self.preset_dropdown.grid(row=1, column=1, padx=5, pady=5, sticky="ew")
        
        self.edit_preset_button = ctk.CTkButton(
            self.preset_frame,
            text="✏️ Éditer",
            command=self._open_preset_editor,
            width=80
        )
        self.edit_preset_button.grid(row=1, column=2, padx=5, pady=5)
        
        self.preset_frame.grid_columnconfigure(1, weight=1)
        
        # ===== ONGLET SYNTHÉTISEUR =====
        # Configuration de la grille pour 4 colonnes dans l'onglet Synthétiseur
        self.tab_synth.grid_rowconfigure(1, weight=1)
        self.tab_synth.grid_columnconfigure((0, 1, 2, 3), weight=1)
        
        # Oscillateur
        self.osc_frame = ctk.CTkFrame(self.tab_synth)
        self.osc_frame.grid(row=1, column=0, padx=5, pady=10, sticky="nsew")
        self._create_oscillator_controls_in_frame()
        
        # Chorus
        self.chorus_main_frame = ctk.CTkFrame(self.tab_synth)
        self.chorus_main_frame.grid(row=1, column=1, padx=5, pady=10, sticky="nsew")
        self._create_chorus_controls_in_frame()
        
        # Reverb
        self.reverb_main_frame = ctk.CTkFrame(self.tab_synth)
        self.reverb_main_frame.grid(row=1, column=2, padx=5, pady=10, sticky="nsew")
        self._create_reverb_controls_in_frame()
        
        # Volume
        self.volume_main_frame = ctk.CTkFrame(self.tab_synth)
        self.volume_main_frame.grid(row=1, column=3, padx=5, pady=10, sticky="nsew")
        self._create_volume_controls_in_frame()
        
        # ===== ONGLET SOUNDBANK =====
        # Configuration de la grille pour l'onglet SoundBank
        self.tab_sf2.grid_rowconfigure(1, weight=1)
        self.tab_sf2.grid_columnconfigure((0, 1, 2), weight=1)
        
        # Frame pour la sélection de SoundFont (en haut)
        self.sf2_frame = ctk.CTkFrame(self.tab_sf2)
        self.sf2_frame.grid(row=0, column=0, columnspan=3, padx=10, pady=10, sticky="ew")
        
        self.sf2_load_button = ctk.CTkButton(
            self.sf2_frame,
            text="📂 Charger une nouvelle SoundFont (.sf2/.sf3)",
            command=self._on_load_sf2,
            height=40,
            font=("Arial", 13, "bold")
        )
        self.sf2_load_button.pack(pady=10, padx=20, fill="x")
        
        # Affichage du nom de la SoundFont
        self.sf2_name_label = ctk.CTkLabel(
            self.sf2_frame,
            text="Aucune SoundFont chargée",
            font=("Arial", 11, "italic"),
            text_color="gray"
        )
        self.sf2_name_label.pack(pady=5)
        
        instr_frame = ctk.CTkFrame(self.sf2_frame, fg_color="transparent")
        instr_frame.pack(pady=10, padx=20, fill="x")
        
        ctk.CTkLabel(
            instr_frame,
            text="Sélecteur d'instrument:",
            font=("Arial", 14, "bold")
        ).pack(side="left", padx=10)
        
        # Label pour afficher l'instrument sélectionné
        self.sf2_instrument_label = ctk.CTkLabel(
            instr_frame,
            text="Aucun instrument",
            font=("Arial", 13),
            anchor="w"
        )
        self.sf2_instrument_label.pack(side="left", padx=10, fill="x", expand=True)
        
        # Bouton pour ouvrir la fenêtre de sélection d'instruments
        self.sf2_select_button = ctk.CTkButton(
            instr_frame,
            text="📋 Sélectionner",
            command=self._open_instrument_selector,
            width=150,
            font=("Arial", 12)
        )
        self.sf2_select_button.pack(side="left", padx=5)

        self.sf2_instruments_data = []

        
        # Contrôles d'effets pour SoundBank (même disposition que Synthétiseur)
        # Chorus
        self.sf2_chorus_frame = ctk.CTkFrame(self.tab_sf2)
        self.sf2_chorus_frame.grid(row=1, column=0, padx=5, pady=10, sticky="nsew")
        self._create_chorus_controls_in_frame(parent_frame=self.sf2_chorus_frame)
        
        # Reverb
        self.sf2_reverb_frame = ctk.CTkFrame(self.tab_sf2)
        self.sf2_reverb_frame.grid(row=1, column=1, padx=5, pady=10, sticky="nsew")
        self._create_reverb_controls_in_frame(parent_frame=self.sf2_reverb_frame)
        
        # Volume
        self.sf2_volume_frame = ctk.CTkFrame(self.tab_sf2)
        self.sf2_volume_frame.grid(row=1, column=2, padx=5, pady=10, sticky="nsew")
        self._create_volume_controls_in_frame(parent_frame=self.sf2_volume_frame)
        
        # ===== ONGLET SÉQUENCEUR =====
        self.sequencer_gui = SequencerGUI(
            self.tab_sequencer, 
            self.sequencer_engine,
            preset_manager=self.preset_manager,
            sf2_instruments_data=self.sf2_instruments_data
        )
        self.sequencer_gui.grid(row=0, column=0, sticky="nsew", padx=0, pady=0)
        
        # Connecter les callbacks du séquenceur aux moteurs de synthèse
        self.sequencer_gui.set_note_callbacks(
            note_on_callback=self._sequencer_note_on,
            note_off_callback=self._sequencer_note_off
        )
        
        # ===== FOOTER (simplifié) =====
        self.footer_frame = ctk.CTkFrame(self, fg_color="transparent")
        self.footer_frame.grid(row=4, column=0, columnspan=4, padx=10, pady=(5, 10), sticky="ew")
        
        self.status_label = ctk.CTkLabel(
            self.footer_frame,
            text="🎹 Synthétiseur MIDI Python",
            font=("Arial", 10)
        )
        self.status_label.pack(pady=5)
    
    def _create_oscillator_controls_in_frame(self):
        """Crée les contrôles de l'oscillateur avec ADSR."""
        ctk.CTkLabel(
            self.osc_frame,
            text="🎹 OSCILLATEUR",
            font=("Arial", 14, "bold")
        ).pack(pady=(10, 10))
        
        # Forme d'onde
        waveform_frame = ctk.CTkFrame(self.osc_frame, fg_color="transparent")
        waveform_frame.pack(pady=5, padx=10)
        
        ctk.CTkLabel(
            waveform_frame,
            text="Forme d'onde:",
            font=("Arial", 12)
        ).pack(side="left", padx=5)
        
        self.waveform_var = ctk.StringVar(value="sawtooth")
        self.waveform_dropdown = ctk.CTkOptionMenu(
            waveform_frame,
            variable=self.waveform_var,
            values=["sine", "square", "sawtooth", "triangle"],
            command=self._on_waveform_changed,
            width=150
        )
        self.waveform_dropdown.pack(side="left", padx=5)
        
        # Séparateur
        ctk.CTkLabel(self.osc_frame, text="", height=10).pack()
        
        # ADSR
        ctk.CTkLabel(
            self.osc_frame,
            text="ADSR",
            font=("Arial", 12, "bold")
        ).pack(pady=(5, 5))
        
        adsr_sliders = ctk.CTkFrame(self.osc_frame, fg_color="transparent")
        adsr_sliders.pack(pady=5, fill="x", padx=10)
        adsr_sliders.grid_columnconfigure((0, 1, 2, 3), weight=1)
        
        # Attack
        self._create_parameter_slider(
            adsr_sliders, "Attack", 0, 0.001, 2.0, 0.01,
            self._on_attack_changed, "s"
        )
        
        # Decay
        self._create_parameter_slider(
            adsr_sliders, "Decay", 1, 0.001, 2.0, 0.1,
            self._on_decay_changed, "s"
        )
        
        # Sustain
        self._create_parameter_slider(
            adsr_sliders, "Sustain", 2, 0.0, 1.0, 0.7,
            self._on_sustain_changed, "%", scale=100
        )
        
        # Release
        self._create_parameter_slider(
            adsr_sliders, "Release", 3, 0.001, 3.0, 0.3,
            self._on_release_changed, "s"
        )
    
    def _create_parameter_slider(self, parent, label, column, min_val, max_val, default, callback, unit, scale=1):
        """Crée un slider vertical pour un paramètre ADSR."""
        frame = ctk.CTkFrame(parent, fg_color="transparent")
        frame.grid(row=0, column=column, padx=10, pady=5)
        
        ctk.CTkLabel(frame, text=label, font=("Arial", 12, "bold")).pack(pady=5)
        
        slider = ctk.CTkSlider(
            frame,
            from_=min_val,
            to=max_val,
            orientation="vertical",
            height=150,
            command=callback
        )
        slider.set(default)
        slider.pack(pady=5)
        
        value_label = ctk.CTkLabel(frame, text=f"{default*scale:.0f}{unit}", font=("Arial", 10))
        value_label.pack(pady=5)
        
        # Stocker le label pour mise à jour
        setattr(self, f"{label.lower()}_label", value_label)
        setattr(self, f"{label.lower()}_slider", slider)
        setattr(self, f"{label.lower()}_unit", unit)
        setattr(self, f"{label.lower()}_scale", scale)
    

    
    
    def _create_chorus_controls_in_frame(self, parent_frame=None):
        """Crée les contrôles du Chorus."""
        if parent_frame is None:
            parent_frame = self.chorus_main_frame
            
        chorus_frame = ctk.CTkFrame(parent_frame)
        chorus_frame.pack(padx=10, pady=10, fill="both", expand=True)
        
        ctk.CTkLabel(
            chorus_frame,
            text="🎵 CHORUS",
            font=("Arial", 16, "bold")
        ).pack(pady=(10, 5))
        
        # Switch ON/OFF
        self.chorus_enabled = ctk.BooleanVar(value=False)
        ctk.CTkSwitch(
            chorus_frame,
            text="Activer",
            variable=self.chorus_enabled,
            command=self._on_chorus_toggle,
            font=("Arial", 12)
        ).pack(pady=5)
        
        # Sliders verticaux en 3 colonnes
        chorus_sliders = ctk.CTkFrame(chorus_frame, fg_color="transparent")
        chorus_sliders.pack(pady=10, fill="x", padx=20)
        chorus_sliders.grid_columnconfigure((0, 1, 2), weight=1)
        
        # Rate
        rate_frame = ctk.CTkFrame(chorus_sliders, fg_color="transparent")
        rate_frame.grid(row=0, column=0, padx=10)
        ctk.CTkLabel(rate_frame, text="Rate", font=("Arial", 12, "bold")).pack(pady=3)
        self.chorus_rate_slider = ctk.CTkSlider(
            rate_frame, from_=0.1, to=5.0,
            orientation="vertical", height=120,
            command=self._on_chorus_rate
        )
        self.chorus_rate_slider.set(1.5)
        self.chorus_rate_slider.pack(pady=5)
        self.chorus_rate_label = ctk.CTkLabel(rate_frame, text="1.5 Hz", font=("Arial", 10))
        self.chorus_rate_label.pack()
        
        # Depth
        depth_frame = ctk.CTkFrame(chorus_sliders, fg_color="transparent")
        depth_frame.grid(row=0, column=1, padx=10)
        ctk.CTkLabel(depth_frame, text="Depth", font=("Arial", 12, "bold")).pack(pady=3)
        self.chorus_depth_slider = ctk.CTkSlider(
            depth_frame, from_=0.0, to=1.0,
            orientation="vertical", height=120,
            command=self._on_chorus_depth
        )
        self.chorus_depth_slider.set(0.5)
        self.chorus_depth_slider.pack(pady=5)
        self.chorus_depth_label = ctk.CTkLabel(depth_frame, text="50%", font=("Arial", 10))
        self.chorus_depth_label.pack()
        
        # Mix
        mix_frame = ctk.CTkFrame(chorus_sliders, fg_color="transparent")
        mix_frame.grid(row=0, column=2, padx=10)
        ctk.CTkLabel(mix_frame, text="Mix", font=("Arial", 12, "bold")).pack(pady=3)
        self.chorus_mix_slider = ctk.CTkSlider(
            mix_frame, from_=0.0, to=1.0,
            orientation="vertical", height=120,
            command=self._on_chorus_mix
        )
        self.chorus_mix_slider.set(0.5)
        self.chorus_mix_slider.pack(pady=5)
        self.chorus_mix_label = ctk.CTkLabel(mix_frame, text="50%", font=("Arial", 10))
        self.chorus_mix_label.pack()
    
    def _create_reverb_controls_in_frame(self, parent_frame=None):
        """Crée les contrôles du Reverb."""
        if parent_frame is None:
            parent_frame = self.reverb_main_frame
            
        reverb_frame = ctk.CTkFrame(parent_frame)
        reverb_frame.pack(padx=10, pady=10, fill="both", expand=True)
        
        ctk.CTkLabel(
            reverb_frame,
            text="🌊 REVERB",
            font=("Arial", 16, "bold")
        ).pack(pady=(10, 5))
        
        # Switch ON/OFF
        self.reverb_enabled = ctk.BooleanVar(value=False)
        ctk.CTkSwitch(
            reverb_frame,
            text="Activer",
            variable=self.reverb_enabled,
            command=self._on_reverb_toggle,
            font=("Arial", 12)
        ).pack(pady=5)
        
        # Sliders verticaux en 3 colonnes
        reverb_sliders = ctk.CTkFrame(reverb_frame, fg_color="transparent")
        reverb_sliders.pack(pady=10, fill="x", padx=20)
        reverb_sliders.grid_columnconfigure((0, 1, 2), weight=1)
        
        # Room Size
        room_frame = ctk.CTkFrame(reverb_sliders, fg_color="transparent")
        room_frame.grid(row=0, column=0, padx=10)
        ctk.CTkLabel(room_frame, text="Room", font=("Arial", 12, "bold")).pack(pady=3)
        self.reverb_room_slider = ctk.CTkSlider(
            room_frame, from_=0.0, to=1.0,
            orientation="vertical", height=120,
            command=self._on_reverb_room
        )
        self.reverb_room_slider.set(0.5)
        self.reverb_room_slider.pack(pady=5)
        self.reverb_room_label = ctk.CTkLabel(room_frame, text="50%", font=("Arial", 10))
        self.reverb_room_label.pack()
        
        # Damping
        damp_frame = ctk.CTkFrame(reverb_sliders, fg_color="transparent")
        damp_frame.grid(row=0, column=1, padx=10)
        ctk.CTkLabel(damp_frame, text="Damp", font=("Arial", 12, "bold")).pack(pady=3)
        self.reverb_damp_slider = ctk.CTkSlider(
            damp_frame, from_=0.0, to=1.0,
            orientation="vertical", height=120,
            command=self._on_reverb_damp
        )
        self.reverb_damp_slider.set(0.5)
        self.reverb_damp_slider.pack(pady=5)
        self.reverb_damp_label = ctk.CTkLabel(damp_frame, text="50%", font=("Arial", 10))
        self.reverb_damp_label.pack()
        
        # Mix
        mix_frame = ctk.CTkFrame(reverb_sliders, fg_color="transparent")
        mix_frame.grid(row=0, column=2, padx=10)
        ctk.CTkLabel(mix_frame, text="Mix", font=("Arial", 12, "bold")).pack(pady=3)
        self.reverb_mix_slider = ctk.CTkSlider(
            mix_frame, from_=0.0, to=1.0,
            orientation="vertical", height=120,
            command=self._on_reverb_mix
        )
        self.reverb_mix_slider.set(0.3)
        self.reverb_mix_slider.pack(pady=5)
        self.reverb_mix_label = ctk.CTkLabel(mix_frame, text="30%", font=("Arial", 10))
        self.reverb_mix_label.pack()

    def _create_volume_controls_in_frame(self, parent_frame=None):
        """Crée les contrôles de volume et PANIC."""
        if parent_frame is None:
            parent_frame = self.volume_main_frame
            
        vol_frame = ctk.CTkFrame(parent_frame)
        vol_frame.pack(padx=10, pady=10, fill="both", expand=True)
        
        ctk.CTkLabel(
            vol_frame,
            text="🔊 VOLUME",
            font=("Arial", 16, "bold")
        ).pack(pady=(10, 5))
        
        # Slider vertical pour le volume
        self.volume_slider = ctk.CTkSlider(
            vol_frame,
            from_=0.0,
            to=1.0,
            orientation="vertical",
            height=150,
            command=self._on_volume_changed
        )
        self.volume_slider.set(0.6)
        self.volume_slider.pack(pady=10)
        
        self.volume_label = ctk.CTkLabel(vol_frame, text="60%", font=("Arial", 12))
        self.volume_label.pack(pady=5)
        
        # Bouton PANIC
        self.panic_button = ctk.CTkButton(
            vol_frame,
            text="⚠️ PANIC",
            command=self._on_panic,
            fg_color="red",
            hover_color="darkred",
            height=40,
            font=("Arial", 13, "bold")
        )
        self.panic_button.pack(pady=10)


    # ===== CALLBACKS =====
    
    def _auto_connect_midi(self):
        """Connexion automatique au port MIDI sauvegardé au démarrage."""
        last_port = self.config_manager.get_last_midi_port()
        if last_port:
            try:
                ports = mido.get_input_names()
                if last_port in ports:
                    self.midi_port_var.set(last_port)
                    self._connect_midi(last_port)
                    print(f"✓ Connexion automatique à {last_port}")
            except Exception as e:
                print(f"Impossible de se connecter automatiquement: {e}")
    
    def _refresh_midi_ports(self):
        """Rafraîchit la liste des ports MIDI."""
        try:
            ports = mido.get_input_names()
            if ports:
                self.midi_dropdown.configure(values=ports)
                # Sélectionner le dernier port utilisé si disponible
                last_port = self.config_manager.get_last_midi_port()
                if last_port and last_port in ports:
                    self.midi_port_var.set(last_port)
                else:
                    self.midi_port_var.set(ports[0])
            else:
                self.midi_dropdown.configure(values=["Aucun port MIDI détecté"])
                self.midi_port_var.set("Aucun port MIDI détecté")
        except Exception as e:
            print(f"Erreur lors de la récupération des ports MIDI: {e}")
    
    def _on_midi_port_selected(self, port_name):
        """Callback quand un port MIDI est sélectionné."""
        if port_name != "Aucun port MIDI détecté":
            self._connect_midi(port_name)
    
    def _on_buffer_size_changed(self, buffer_size):
        """Callback quand la taille du buffer est changée."""
        # Sauvegarder la préférence
        self.config_manager.config['buffer_size'] = int(buffer_size)
        self.config_manager.save_config()
        
        # Redémarrer l'audio si déjà connecté
        if self.is_running:
            current_port = self.midi_port_var.get()
            print(f"🔄 Redémarrage audio avec buffer {buffer_size}...")
            self._disconnect_midi()
            self._connect_midi(current_port)
    
    def _connect_midi(self, port_name):
        """Connecte au port MIDI et démarre l'audio."""
        # Arrêter la connexion existante si présente
        if self.is_running:
            self._disconnect_midi()
        
        try:
            # Démarrer l'audio si nécessaire
            if self.audio_output:
                if not self.audio_output.is_running:
                    # Mettre à jour le buffer size depuis l'UI avant de démarrer
                    try:
                        self.audio_output.buffer_size = int(self.buffer_size_var.get())
                    except:
                        pass
                    self.audio_output.start()
            
            # Démarrer le thread MIDI
            self.is_running = True
            self.midi_thread = threading.Thread(target=self._midi_loop, args=(port_name,), daemon=True)
            self.midi_thread.start()
            
            # Mettre à jour l'interface
            self.connection_indicator.configure(text="● Connecté", text_color="green")
            
            # Sauvegarder le port
            self.config_manager.set_last_midi_port(port_name)
            
            print(f"✓ Connecté à {port_name}")
            
        except Exception as e:
            print(f"Erreur de connexion MIDI: {e}")
            self.connection_indicator.configure(text="● Erreur", text_color="orange")
    
    def _disconnect_midi(self):
        """Déconnecte le MIDI."""
        self.is_running = False
        if self.midi_thread:
            self.midi_thread.join(timeout=1.0)
        self.connection_indicator.configure(text="● Déconnecté", text_color="red")
    
    def _midi_loop(self, port_name):
        """Boucle de réception MIDI."""
        try:
            with mido.open_input(port_name) as inport:
                for msg in inport:
                    if not self.is_running:
                        break
                    
                    if msg.type == 'note_on' and msg.velocity > 0:
                        # Si le séquenceur est en enregistrement OU si une piste est armée (monitoring)
                        if self.sequencer_engine.state.value == "recording" or self.sequencer_engine.get_armed_track():
                            self.sequencer_engine.record_note_on(msg.note, msg.velocity)
                        else:
                            # Sinon, jouer en direct sur le moteur global
                            if self.current_engine:
                                self.current_engine.note_on(msg.note, msg.velocity)
                            
                    elif msg.type == 'note_off' or (msg.type == 'note_on' and msg.velocity == 0):
                        # Monitoring ou enregistrement
                        if self.sequencer_engine.state.value == "recording" or self.sequencer_engine.get_armed_track():
                            self.sequencer_engine.record_note_off(msg.note)
                        else:
                            # Sinon, jouer en direct sur le moteur global
                            if self.current_engine:
                                self.current_engine.note_off(msg.note)
        except Exception as e:
            print(f"Erreur dans la boucle MIDI: {e}")
            self.is_running = False
    
    def _on_category_selected(self, category):
        """Callback quand une catégorie est sélectionnée."""
        # Mettre à jour la liste des presets
        preset_names = self.preset_manager.get_presets_in_category(category)
        if preset_names:
            self.preset_dropdown.configure(values=preset_names)
            # Sélectionner le premier preset de la catégorie
            self.preset_var.set(preset_names[0])
            self._load_preset(category, preset_names[0])
        else:
            self.preset_dropdown.configure(values=["Aucun preset"])
            self.preset_var.set("Aucun preset")
        
        # Sauvegarder la catégorie
        self.config_manager.set_last_category(category)
    
    def _on_preset_selected(self, preset_name):
        """Callback quand un preset est sélectionné - charge immédiatement."""
        category = self.category_var.get()
        self._load_preset(category, preset_name)
    
    def _load_preset(self, category, preset_name):
        """Charge un preset."""
        if self.preset_manager.apply_preset(self.synth_engine, category, preset_name):
            # Mettre à jour l'interface pour refléter les paramètres du preset
            preset = self.preset_manager.get_preset(category, preset_name)
            if preset:
                # Waveform
                self.waveform_var.set(preset.get('waveform', 'sine'))
                
                # ADSR
                self.attack_slider.set(preset.get('attack', 0.01))
                self.decay_slider.set(preset.get('decay', 0.1))
                self.sustain_slider.set(preset.get('sustain', 0.7))
                self.release_slider.set(preset.get('release', 0.3))
                
                # Volume
                self.volume_slider.set(preset.get('master_volume', 0.6))
                
                # Mettre à jour les labels
                self._update_all_labels()
                
                # Sauvegarder
                self.config_manager.set_last_category(category)
                self.config_manager.set_last_preset(preset_name)
                print(f"✓ Preset '{preset_name}' chargé")
    
    
    def _update_all_labels(self):
        """Met à jour tous les labels de valeurs."""
        # ADSR
        self.attack_label.configure(text=f"{self.attack_slider.get()*1000:.0f}ms")
        self.decay_label.configure(text=f"{self.decay_slider.get()*1000:.0f}ms")
        self.sustain_label.configure(text=f"{self.sustain_slider.get()*100:.0f}%")
        self.release_label.configure(text=f"{self.release_slider.get()*1000:.0f}ms")
        # Volume
        self.volume_label.configure(text=f"{self.volume_slider.get()*100:.0f}%")
    
    def _on_waveform_changed(self, waveform):
        """Callback changement de forme d'onde."""
        self.synth_engine.set_waveform(waveform)
    
    def _on_attack_changed(self, value):
        """Callback changement Attack."""
        self.synth_engine.set_adsr(
            attack=value,
            decay=self.decay_slider.get(),
            sustain=self.sustain_slider.get(),
            release=self.release_slider.get()
        )
        self.attack_label.configure(text=f"{value*1000:.0f}ms")
    
    def _on_decay_changed(self, value):
        """Callback changement Decay."""
        self.synth_engine.set_adsr(
            attack=self.attack_slider.get(),
            decay=value,
            sustain=self.sustain_slider.get(),
            release=self.release_slider.get()
        )
        self.decay_label.configure(text=f"{value*1000:.0f}ms")
    
    def _on_sustain_changed(self, value):
        """Callback changement Sustain."""
        self.synth_engine.set_adsr(
            attack=self.attack_slider.get(),
            decay=self.decay_slider.get(),
            sustain=value,
            release=self.release_slider.get()
        )
        self.sustain_label.configure(text=f"{value*100:.0f}%")
    
    def _on_release_changed(self, value):
        """Callback changement Release."""
        self.synth_engine.set_adsr(
            attack=self.attack_slider.get(),
            decay=self.decay_slider.get(),
            sustain=self.sustain_slider.get(),
            release=value
        )
        self.release_label.configure(text=f"{value*1000:.0f}ms")
    
    def _on_cutoff_changed(self, value):
        """Callback changement Cutoff."""
        self.synth_engine.set_filter(cutoff=value, resonance=self.resonance_slider.get())
        self.cutoff_label.configure(text=f"{value:.0f} Hz")
    
    def _on_resonance_changed(self, value):
        """Callback changement Resonance."""
        self.synth_engine.set_filter(cutoff=self.cutoff_slider.get(), resonance=value)
        self.resonance_label.configure(text=f"{value:.1f}")
    
    def _on_volume_changed(self, value):
        """Callback changement Volume."""
        self.synth_engine.set_master_volume(value)
        self.sf2_engine.set_master_volume(value)
        self.volume_label.configure(text=f"{value*100:.0f}%")
    
    def _on_tab_changed(self):
        """Bascule le moteur de synthèse en fonction de l'onglet sélectionné."""
        current_tab = self.tabview.get()
        if current_tab == "Synthétiseur":
            mode = "Synthétiseur (Soustractif)"
        else:
            mode = "Banque de sons (SoundFont)"
        
        self._on_engine_mode_changed(mode)

    def _on_engine_mode_changed(self, mode):
        """Bascule entre le synthé soustractif et le moteur SoundFont."""
        if mode == "Synthétiseur (Soustractif)":
            self.current_engine = self.synth_engine
            self.sf2_load_button.configure(state="disabled")
            self.sf2_select_button.configure(state="disabled")
            # Réactiver les contrôles ADSR/Waveform
            self.osc_frame.configure(fg_color="transparent")
        else:
            self.current_engine = self.sf2_engine
            self.sf2_load_button.configure(state="normal")
            if self.sf2_instruments_data:
                self.sf2_select_button.configure(state="normal")
            # On pourrait griser l'OSC frame pour indiquer qu'elle n'est plus active
            self.osc_frame.configure(fg_color="gray30")
        
        # Sauvegarder le mode de moteur
        self.config_manager.set_engine_mode(mode)
        
        print(f"Moteur de synthèse : {mode}")

    def _on_load_sf2(self):
        """Ouvre un dialogue pour charger un fichier SoundFont (.sf2 ou .sf3)."""
        from tkinter import filedialog
        import os
        file_path = filedialog.askopenfilename(
            title="Charger un SoundFont",
            filetypes=[("SoundFont files", "*.sf2 *.sf3"), ("All files", "*.*")]
        )
        if file_path:
            if self.sf2_engine.load_soundfont(file_path):
                # Afficher le nom et le chemin du fichier SoundFont
                sf2_name = os.path.basename(file_path)
                self.sf2_name_label.configure(
                    text=f"🎵 {sf2_name}\n{file_path}",
                    text_color="#3b8ed0"  # Couleur bleue
                )
                print(f"✓ SoundFont chargé : {file_path}")
                # Sauvegarder le chemin de la SoundFont
                self.config_manager.set_soundfont_path(file_path)
                self._refresh_sf2_instruments()
            else:
                from tkinter import messagebox
                messagebox.showerror("Erreur", f"Impossible de charger le SoundFont :\n{file_path}")
    
    def _refresh_sf2_instruments(self, auto_select=True):
        """Met à jour la liste des instruments à partir du moteur SF2.
        
        Args:
            auto_select: Si True, sélectionne automatiquement le premier instrument
        """
        self.sf2_instruments_data = self.sf2_engine.get_available_instruments()
        
        # Mettre à jour le séquenceur si nécessaire
        if self.sequencer_gui:
            self.sequencer_gui.sf2_instruments_data = self.sf2_instruments_data
        
        if self.sf2_instruments_data:
            # Activer le bouton de sélection
            self.sf2_select_button.configure(state="normal")
            # Sélectionner automatiquement le premier instrument (seulement si demandé)
            if auto_select:
                first_instrument = self.sf2_instruments_data[0]['name']
                self._on_sf2_instrument_selected(first_instrument)
        else:
            # Désactiver le bouton de sélection
            self.sf2_select_button.configure(state="disabled")
            self.sf2_instrument_label.configure(text="Aucun instrument")

    def _on_sf2_instrument_selected(self, instrument_name):
        """Callback quand un instrument SF2 est sélectionné."""
        # Trouver l'instrument correspondant dans les données
        for inst in self.sf2_instruments_data:
            if inst['name'] == instrument_name:
                self.sf2_engine.set_instrument(inst['bank'], inst['preset'])
                self.sf2_instrument_label.configure(text=instrument_name)
                print(f"Instrument SF2 sélectionné : {instrument_name} (Bank: {inst['bank']}, Preset: {inst['preset']})")
                # Sauvegarder l'instrument sélectionné
                self.config_manager.set_soundfont_instrument(instrument_name, inst['bank'], inst['preset'])
                break
    
    def _open_instrument_selector(self):
        """Ouvre une fenêtre popup pour sélectionner un instrument."""
        if not self.sf2_instruments_data:
            from tkinter import messagebox
            messagebox.showinfo("Information", "Aucun instrument disponible. Veuillez charger une SoundFont d'abord.")
            return
        
        # Créer une fenêtre Toplevel
        selector_window = ctk.CTkToplevel(self)
        selector_window.title("Sélection d'instrument")
        selector_window.geometry("900x600")
        selector_window.transient(self)  # La rendre modale par rapport à la fenêtre principale
        
        # Attendre que la fenêtre soit visible avant d'appeler grab_set
        selector_window.update_idletasks()
        selector_window.after(50, selector_window.grab_set)  # Bloquer l'interaction avec la fenêtre principale
        
        # Frame du haut pour le titre et les contrôles
        header_frame = ctk.CTkFrame(selector_window, fg_color="transparent")
        header_frame.pack(pady=10, padx=10, fill="x")
        
        # Titre
        title_label = ctk.CTkLabel(
            header_frame,
            text="Sélectionnez un instrument",
            font=("Arial", 16, "bold")
        )
        title_label.pack(side="left", padx=10)
        
        # Contrôle pour le nombre d'instruments par ligne
        config_frame = ctk.CTkFrame(header_frame, fg_color="transparent")
        config_frame.pack(side="right", padx=10)
        
        ctk.CTkLabel(
            config_frame,
            text="Par ligne:",
            font=("Arial", 11)
        ).pack(side="left", padx=5)
        
        # Récupérer la valeur actuelle
        current_per_row = self.config_manager.get_instruments_per_row()
        
        # Variable pour le nombre d'instruments par ligne
        per_row_var = ctk.IntVar(value=current_per_row)
        
        # Label pour afficher le nombre
        per_row_label = ctk.CTkLabel(
            config_frame,
            text=str(current_per_row),
            font=("Arial", 12, "bold"),
            width=30
        )
        per_row_label.pack(side="left", padx=5)
        
        # Fonction pour mettre à jour l'affichage quand on change le nombre par ligne
        def update_grid():
            # Sauvegarder la nouvelle valeur
            new_value = per_row_var.get()
            self.config_manager.set_instruments_per_row(new_value)
            self.config_manager.save_config()
            per_row_label.configure(text=str(new_value))
            # Rafraîchir la grille
            refresh_instrument_grid()
        
        def decrease_per_row():
            current = per_row_var.get()
            if current > 1:
                per_row_var.set(current - 1)
                update_grid()
        
        def increase_per_row():
            current = per_row_var.get()
            if current < 20:
                per_row_var.set(current + 1)
                update_grid()
        
        # Bouton -1
        minus_btn = ctk.CTkButton(
            config_frame,
            text="-1",
            command=decrease_per_row,
            width=40,
            font=("Arial", 12, "bold")
        )
        minus_btn.pack(side="left", padx=2)
        
        # Bouton +1
        plus_btn = ctk.CTkButton(
            config_frame,
            text="+1",
            command=increase_per_row,
            width=40,
            font=("Arial", 12, "bold")
        )
        plus_btn.pack(side="left", padx=2)
        
        # Frame pour la recherche
        search_frame = ctk.CTkFrame(selector_window, fg_color="transparent")
        search_frame.pack(pady=5, padx=10, fill="x")
        
        ctk.CTkLabel(
            search_frame,
            text="🔍 Recherche (tags séparés par espaces):",
            font=("Arial", 11)
        ).pack(side="left", padx=5)
        
        # Variable pour le champ de recherche
        search_var = ctk.StringVar()
        
        # Fonction pour filtrer les instruments
        def on_search_change(*args):
            refresh_instrument_grid()
        
        search_var.trace_add("write", on_search_change)
        
        search_entry = ctk.CTkEntry(
            search_frame,
            textvariable=search_var,
            placeholder_text="Ex: org ha",
            width=300,
            font=("Arial", 11)
        )
        search_entry.pack(side="left", padx=5, fill="x", expand=True)
        
        # Bouton pour effacer la recherche
        clear_btn = ctk.CTkButton(
            search_frame,
            text="✕",
            command=lambda: search_var.set(""),
            width=30,
            font=("Arial", 12, "bold")
        )
        clear_btn.pack(side="left", padx=2)
        
        # Frame scrollable pour les boutons en grille
        scrollable_frame = ctk.CTkScrollableFrame(
            selector_window,
            width=850,
            height=400
        )
        scrollable_frame.pack(pady=10, padx=10, fill="both", expand=True)
        
        # Fonction pour créer la grille d'instruments
        def refresh_instrument_grid():
            # Détruire tous les widgets existants
            for widget in scrollable_frame.winfo_children():
                widget.destroy()
            
            # Récupérer le nombre d'instruments par ligne
            instruments_per_row = per_row_var.get()
            
            # Filtrer les instruments selon la recherche
            search_text = search_var.get().strip().lower()
            tags = search_text.split() if search_text else []
            
            filtered_instruments = []
            for inst in self.sf2_instruments_data:
                inst_name_lower = inst['name'].lower()
                # Tous les tags doivent être présents dans le nom
                if all(tag in inst_name_lower for tag in tags):
                    filtered_instruments.append(inst)
            
            # Créer un bouton pour chaque instrument
            def select_instrument(inst_name):
                self._on_sf2_instrument_selected(inst_name)
                selector_window.destroy()
            
            row = 0
            col = 0
            
            for inst in filtered_instruments:
                inst_name = inst['name']
                
                # Calculer la largeur du bouton en fonction du nombre par ligne
                # Largeur totale ~850px, moins les marges
                button_width = max(100, (850 - (instruments_per_row + 1) * 10) // instruments_per_row)
                
                btn = ctk.CTkButton(
                    scrollable_frame,
                    text=inst_name,
                    command=lambda name=inst_name: select_instrument(name),
                    height=50,
                    width=button_width,
                    font=("Arial", 11),
                    anchor="center"
                )
                btn.grid(row=row, column=col, padx=5, pady=5, sticky="ew")
                
                col += 1
                if col >= instruments_per_row:
                    col = 0
                    row += 1
            
            # Configurer les colonnes pour qu'elles aient toutes le même poids
            for i in range(instruments_per_row):
                scrollable_frame.grid_columnconfigure(i, weight=1)
        
        # Initialiser la grille
        refresh_instrument_grid()
        
        # Bouton Annuler
        cancel_btn = ctk.CTkButton(
            selector_window,
            text="Annuler",
            command=selector_window.destroy,
            fg_color="gray",
            hover_color="darkgray"
        )
        cancel_btn.pack(pady=10)

    def _load_default_sf2(self):
        """Tente de charger la SoundFont FluidR3_GM par défaut."""
        import os
        default_sf2 = "/usr/share/sounds/sf2/FluidR3_GM.sf2"
        if os.path.exists(default_sf2):
            if self.sf2_engine.load_soundfont(default_sf2):
                # Afficher le nom et le chemin du fichier SoundFont
                sf2_name = os.path.basename(default_sf2)
                self.sf2_name_label.configure(
                    text=f"🎵 {sf2_name}\n{default_sf2}",
                    text_color="#3b8ed0"  # Couleur bleue
                )
                print(f"✓ SoundFont par défaut chargée : {default_sf2}")
                # Sauvegarder le chemin de la SoundFont par défaut (seulement si aucune SoundFont n'est déjà configurée)
                if not self.config_manager.get_soundfont_path():
                    self.config_manager.set_soundfont_path(default_sf2)
                # Ne pas auto-sélectionner ici, la restauration s'en chargera
                self._refresh_sf2_instruments(auto_select=False)
            else:
                print(f"⚠️ Échec du chargement de la SoundFont par défaut : {default_sf2}")
        else:
            print(f"⚠️ SoundFont par défaut introuvable : {default_sf2}")
    
    def _restore_soundfont_config(self):
        """Restaure la configuration SoundFont sauvegardée (moteur, SoundFont, instrument)."""
        import os
        
        # Restaurer le moteur actif
        saved_mode = self.config_manager.get_engine_mode()
        if saved_mode and saved_mode != "Synthétiseur (Soustractif)":
            # Basculer vers l'onglet SoundBank
            self.tabview.set("SoundBank")
            # Forcer l'appel du callback car .set() ne le fait pas forcément
            self._on_tab_changed()
        
        # Restaurer la SoundFont sauvegardée
        saved_sf2_path = self.config_manager.get_soundfont_path()
        if saved_sf2_path and os.path.exists(saved_sf2_path):
            # Charger la SoundFont sauvegardée
            if self.sf2_engine.load_soundfont(saved_sf2_path):
                sf2_name = os.path.basename(saved_sf2_path)
                self.sf2_name_label.configure(
                    text=f"🎵 {sf2_name}\n{saved_sf2_path}",
                    text_color="#3b8ed0"
                )
                print(f"✓ SoundFont restaurée : {saved_sf2_path}")
                # Rafraîchir la liste sans auto-sélectionner
                self._refresh_sf2_instruments(auto_select=False)
                
                # Restaurer l'instrument sauvegardé
                saved_instrument = self.config_manager.get_soundfont_instrument()
                restored = False
                if saved_instrument and saved_instrument.get("name"):
                    # Vérifier que l'instrument existe dans la SoundFont
                    for inst in self.sf2_instruments_data:
                        if (inst['name'] == saved_instrument['name'] and 
                            inst['bank'] == saved_instrument['bank'] and 
                            inst['preset'] == saved_instrument['preset']):
                            # Sélectionner l'instrument sauvegardé
                            self._on_sf2_instrument_selected(saved_instrument['name'])
                            print(f"✓ Instrument restauré : {saved_instrument['name']}")
                            restored = True
                            break
                
                # Si non restauré, sélectionner le premier par défaut
                if not restored and self.sf2_instruments_data:
                    first_instrument = self.sf2_instruments_data[0]['name']
                    self._on_sf2_instrument_selected(first_instrument)
            else:
                print(f"⚠️ Impossible de charger la SoundFont sauvegardée : {saved_sf2_path}")
                # En cas d'échec, on s'assure qu'un instrument est sélectionné dans la SoundFont par défaut
                if self.sf2_instruments_data:
                    self._on_sf2_instrument_selected(self.sf2_instruments_data[0]['name'])
        else:
            # Si aucune SoundFont n'est restaurée, on sélectionne le premier instrument de la SoundFont par défaut
            if self.sf2_instruments_data:
                # On ne sélectionne que si aucun instrument n'est déjà actif (évite les doubles appels)
                current_text = self.sf2_instrument_label.cget("text")
                if current_text == "Aucun instrument sélectionné":
                    self._on_sf2_instrument_selected(self.sf2_instruments_data[0]['name'])
    
    def _on_panic(self):
        """Callback bouton PANIC."""
        self.synth_engine.panic()
        print("⚠️ PANIC - Toutes les notes arrêtées")
    
    def _on_chorus_toggle(self):
        """Callback toggle Chorus."""
        self.synth_engine.chorus.set_enabled(self.chorus_enabled.get())
        print(f"🎵 Chorus: {'ON' if self.chorus_enabled.get() else 'OFF'}")
    
    def _on_chorus_rate(self, value):
        """Callback rate Chorus."""
        self.synth_engine.chorus.set_rate(value)
        self.chorus_rate_label.configure(text=f"{value:.1f} Hz")
    
    def _on_chorus_depth(self, value):
        """Callback depth Chorus."""
        self.synth_engine.chorus.set_depth(value)
        self.chorus_depth_label.configure(text=f"{value*100:.0f}%")
    
    def _on_chorus_mix(self, value):
        """Callback mix Chorus."""
        self.synth_engine.chorus.set_mix(value)
        self.chorus_mix_label.configure(text=f"{value*100:.0f}%")
    
    def _on_reverb_toggle(self):
        """Callback toggle Reverb."""
        self.synth_engine.reverb.set_enabled(self.reverb_enabled.get())
        print(f"🌊 Reverb: {'ON' if self.reverb_enabled.get() else 'OFF'}")
    
    def _on_reverb_room(self, value):
        """Callback room size Reverb."""
        self.synth_engine.reverb.set_room_size(value)
        self.reverb_room_label.configure(text=f"{value*100:.0f}%")
    
    def _on_reverb_damp(self, value):
        """Callback damping Reverb."""
        self.synth_engine.reverb.set_damping(value)
        self.reverb_damp_label.configure(text=f"{value*100:.0f}%")
    
    def _on_reverb_mix(self, value):
        """Callback mix Reverb."""
        self.synth_engine.reverb.set_mix(value)
        self.reverb_mix_label.configure(text=f"{value*100:.0f}%")
    
    def _open_preset_editor(self):
        """Ouvre la fenêtre d'édition de presets."""
        current_category = self.category_var.get()
        PresetEditorWindow(self, self.preset_manager, self.synth_engine, 
                          self.category_var, self.preset_var)
    
    def _start_status_updates(self):
        """Démarre le thread de mise à jour du status."""
        self.status_running = True
        self.status_thread = threading.Thread(target=self._status_update_loop, daemon=True)
        self.status_thread.start()
    
    def _status_update_loop(self):
        """Boucle de mise à jour du status (ex: tous les 100ms)."""
        import time
        while self.status_running:
            try:
                # Mettre à jour le status dans le thread principal (thread-safe)
                self.after(0, self._update_status_display)
                time.sleep(0.1)  # Mise à jour 10 fois par seconde
            except Exception as e:
                print(f"Erreur dans status update: {e}")
    
    def _update_status_display(self):
        """Met à jour l'affichage du status (appelé depuis le thread principal)."""
        # Footer simplifié - pas de mise à jour nécessaire
        pass
    
    def _on_closing(self):
        """Callback fermeture de la fenêtre."""
        # Arrêter le thread de status
        self.status_running = False
        if self.status_thread:
            self.status_thread.join(timeout=0.5)
        
        # Sauvegarder la position et taille de la fenêtre
        geometry = self.geometry().split('+')
        size = geometry[0].split('x')
        pos = geometry[1:] if len(geometry) > 1 else [100, 100]
        
        self.config_manager.set_window_size(int(size[0]), int(size[1]))
        if len(pos) >= 2:
            self.config_manager.set_window_position(int(pos[0]), int(pos[1]))
        
        # Sauvegarder la configuration
        self.config_manager.save_config()
        
        # Arrêter le MIDI et l'audio
        self._disconnect_midi()
        if self.audio_output:
            self.audio_output.stop()
        
        # Fermer la fenêtre
        self.destroy()
    
    # ===== CALLBACKS SÉQUENCEUR =====
    
    def _sequencer_note_on(self, note: int, velocity: int, track_idx: int = 0):
        """Callback envoyé par le séquenceur pour jouer une note."""
        if track_idx >= 0 and track_idx < len(self.sequencer_engine.tracks):
            track = self.sequencer_engine.tracks[track_idx]
            
            if track.instrument_type == "sf2":
                # Instrument SoundFont
                if self.sf2_engine:
                    self.sf2_engine.set_instrument(track.bank, track.preset)
                    self.sf2_engine.note_on(note, velocity)
            else:
                # Instrument Synthé Soustractif
                if self.synth_engine:
                    # Optimisation : n'appliquer que si l'instrument a changé
                    preset_id = (getattr(track, 'instrument_category', None), track.instrument_name)
                    if preset_id != self.last_applied_synth_preset:
                        if preset_id[0] and preset_id[1]:
                            self.preset_manager.apply_preset(
                                self.synth_engine,
                                preset_id[0],
                                preset_id[1]
                            )
                            self.last_applied_synth_preset = preset_id
                    
                    self.synth_engine.note_on(note, velocity)
        else:
            # Fallback (live play ou pas de piste spécifique)
            if self.current_engine:
                self.current_engine.note_on(note, velocity)
        
    def _sequencer_note_off(self, note: int, track_idx: int = 0):
        """Callback envoyé par le séquenceur pour arrêter une note."""
        if track_idx >= 0 and track_idx < len(self.sequencer_engine.tracks):
            track = self.sequencer_engine.tracks[track_idx]
            if track.instrument_type == "sf2" and self.sf2_engine:
                self.sf2_engine.note_off(note)
            elif self.synth_engine:
                self.synth_engine.note_off(note)
        else:
            if self.current_engine:
                self.current_engine.note_off(note)
    
    # ===== CALLBACKS MENU PROJET =====
    
    def _on_new_project(self):
        """Callback nouveau projet."""
        from tkinter import messagebox
        if messagebox.askyesno("Nouveau projet", "Créer un nouveau projet ? (Les modifications non sauvegardées seront perdues)"):
            # Arrêter la lecture
            self.sequencer_engine.stop()
            # Réinitialiser toutes les pistes
            for track in self.sequencer_engine.tracks:
                track.clear()
            self.sequencer_engine.set_position(0.0)
            # Rafraîchir l'interface
            if self.sequencer_gui:
                self.sequencer_gui._refresh_track_list()
            print("✓ Nouveau projet créé")
    
    def _on_open_project(self):
        """Callback ouvrir projet."""
        from tkinter import filedialog
        from project_manager import ProjectManager
        
        filepath = filedialog.askopenfilename(
            title="Ouvrir un projet",
            initialdir="projects",
            filetypes=[("Fichiers projet", "*.project.json"), ("Tous les fichiers", "*.*")]
        )
        
        if filepath:
            pm = ProjectManager()
            if pm.load_project(filepath, self.sequencer_engine):
                # Rafraîchir l'interface
                if self.sequencer_gui:
                    self.sequencer_gui._refresh_track_list()
                    self.sequencer_gui.bpm_var.set(str(self.sequencer_engine.bpm))
                print(f"✓ Projet chargé : {filepath}")
    
    def _on_save_project(self):
        """Callback sauvegarder projet."""
        from tkinter import filedialog, simpledialog
        from project_manager import ProjectManager
        
        # Demander le nom du projet
        project_name = simpledialog.askstring("Sauvegarder", "Nom du projet:")
        if not project_name:
            return
        
        # Créer le nom de fichier
        filename = f"{project_name}.project.json"
        
        pm = ProjectManager()
        if pm.save_project(self.sequencer_engine, filename, project_name):
            print(f"✓ Projet sauvegardé : {filename}")
    
    def _on_export_midi(self):
        """Callback exporter MIDI."""
        from tkinter import filedialog
        from project_manager import ProjectManager
        
        filepath = filedialog.asksaveasfilename(
            title="Exporter en MIDI",
            defaultextension=".mid",
            filetypes=[("Fichiers MIDI", "*.mid"), ("Tous les fichiers", "*.*")]
        )
        
        if filepath:
            pm = ProjectManager()
            if pm.export_midi(self.sequencer_engine, filepath):
                print(f"✓ Exporté en MIDI : {filepath}")

def main():
    """Point d'entrée de l'interface graphique."""
    app = SynthGUI()
    app.mainloop()

if __name__ == "__main__":
    main()
