"""
Module de génération de formes d'onde pour synthèse audio.
Fournit des fonctions optimisées pour générer les formes d'onde standard.
"""

import numpy as np

def generate_sine(frequency, duration, sample_rate=44100, phase=0.0):
    """
    Génère une forme d'onde sinusoïdale.
    
    Args:
        frequency: Fréquence en Hz
        duration: Durée en secondes
        sample_rate: Taux d'échantillonnage (défaut: 44100 Hz)
        phase: Phase initiale en radians (défaut: 0.0)
    
    Returns:
        numpy array contenant les échantillons audio
    """
    t = np.linspace(0, duration, int(sample_rate * duration), endpoint=False)
    return np.sin(2 * np.pi * frequency * t + phase)

def generate_square(frequency, duration, sample_rate=44100, phase=0.0, duty_cycle=0.5):
    """
    Génère une forme d'onde carrée.
    
    Args:
        frequency: Fréquence en Hz
        duration: Durée en secondes
        sample_rate: Taux d'échantillonnage (défaut: 44100 Hz)
        phase: Phase initiale en radians (défaut: 0.0)
        duty_cycle: Rapport cyclique (défaut: 0.5 pour onde carrée parfaite)
    
    Returns:
        numpy array contenant les échantillons audio
    """
    t = np.linspace(0, duration, int(sample_rate * duration), endpoint=False)
    sine_wave = np.sin(2 * np.pi * frequency * t + phase)
    return np.where(sine_wave >= 2 * duty_cycle - 1, 1.0, -1.0)

def generate_sawtooth(frequency, duration, sample_rate=44100, phase=0.0):
    """
    Génère une forme d'onde en dent de scie.
    
    Args:
        frequency: Fréquence en Hz
        duration: Durée en secondes
        sample_rate: Taux d'échantillonnage (défaut: 44100 Hz)
        phase: Phase initiale en radians (défaut: 0.0)
    
    Returns:
        numpy array contenant les échantillons audio
    """
    t = np.linspace(0, duration, int(sample_rate * duration), endpoint=False)
    # Calcul de la phase combinée
    phase_t = 2 * np.pi * frequency * t + phase
    # Ramener dans l'intervalle [0, 2π]
    phase_t = np.mod(phase_t, 2 * np.pi)
    # Convertir en dent de scie [-1, 1]
    return 2 * (phase_t / (2 * np.pi)) - 1

def generate_triangle(frequency, duration, sample_rate=44100, phase=0.0):
    """
    Génère une forme d'onde triangulaire.
    
    Args:
        frequency: Fréquence en Hz
        duration: Durée en secondes
        sample_rate: Taux d'échantillonnage (défaut: 44100 Hz)
        phase: Phase initiale en radians (défaut: 0.0)
    
    Returns:
        numpy array contenant les échantillons audio
    """
    t = np.linspace(0, duration, int(sample_rate * duration), endpoint=False)
    # Utiliser la dent de scie pour créer un triangle
    phase_t = 2 * np.pi * frequency * t + phase
    phase_t = np.mod(phase_t, 2 * np.pi)
    saw = 2 * (phase_t / (2 * np.pi)) - 1
    # Transformer en triangle par valeur absolue et mise à l'échelle
    return 2 * np.abs(saw) - 1

class Oscillator:
    """
    Oscillateur temps réel pour génération de formes d'onde.
    Maintient la phase entre les appels pour un son continu.
    """
    
    WAVEFORMS = {
        'sine': 0,
        'square': 1,
        'sawtooth': 2,
        'triangle': 3
    }
    
    def __init__(self, sample_rate=44100):
        """
        Initialise l'oscillateur.
        
        Args:
            sample_rate: Taux d'échantillonnage
        """
        self.sample_rate = sample_rate
        self.phase = 0.0
        self.frequency = 440.0  # La (A4) par défaut
        self.waveform = 'sine'
        self.duty_cycle = 0.5
    
    def set_frequency(self, frequency):
        """Définit la fréquence de l'oscillateur."""
        self.frequency = frequency
    
    def set_waveform(self, waveform):
        """
        Définit la forme d'onde.
        
        Args:
            waveform: 'sine', 'square', 'sawtooth', ou 'triangle'
        """
        if waveform in self.WAVEFORMS:
            self.waveform = waveform
    
    def reset_phase(self):
        """Réinitialise la phase à 0."""
        self.phase = 0.0
    
    def generate(self, num_samples):
        """
        Génère un nombre d'échantillons pour la forme d'onde courante.
        
        Args:
            num_samples: Nombre d'échantillons à générer
        
        Returns:
            numpy array des échantillons générés
        """
        # Créer le tableau de temps
        t = np.arange(num_samples) / self.sample_rate
        
        # Calculer la phase pour chaque échantillon
        phase_increment = 2 * np.pi * self.frequency / self.sample_rate
        phases = self.phase + phase_increment * np.arange(num_samples)
        
        # Générer la forme d'onde appropriée
        if self.waveform == 'sine':
            output = np.sin(phases)
        elif self.waveform == 'square':
            sine_wave = np.sin(phases)
            output = np.where(sine_wave >= 2 * self.duty_cycle - 1, 1.0, -1.0)
        elif self.waveform == 'sawtooth':
            normalized_phase = np.mod(phases, 2 * np.pi)
            output = 2 * (normalized_phase / (2 * np.pi)) - 1
        elif self.waveform == 'triangle':
            normalized_phase = np.mod(phases, 2 * np.pi)
            saw = 2 * (normalized_phase / (2 * np.pi)) - 1
            output = 2 * np.abs(saw) - 1
        else:
            output = np.zeros(num_samples)
        
        # Mettre à jour la phase pour la continuité
        self.phase = phases[-1] + phase_increment
        # Garder la phase dans [0, 2π] pour éviter les débordements
        self.phase = np.mod(self.phase, 2 * np.pi)
        
        return output

def midi_to_frequency(midi_note):
    """
    Convertit un numéro de note MIDI en fréquence.
    
    Args:
        midi_note: Numéro MIDI (0-127), où 69 = La 440Hz
    
    Returns:
        Fréquence en Hz
    """
    return 440.0 * (2.0 ** ((midi_note - 69) / 12.0))
