import pygame
import sys
from utils import WINDOW_WIDTH, WINDOW_HEIGHT, BG_COLOR, save_highscore, DIFFICULTIES, load_highscores
from snake import Snake
from food import Food
from ui import UI

class Game:
    def __init__(self):
        pygame.init()
        self.screen = pygame.display.set_mode((WINDOW_WIDTH, WINDOW_HEIGHT))
        pygame.display.set_caption("Neo-Snake")
        self.clock = pygame.time.Clock()
        
        self.state = "MENU" # MENU, PLAYING, GAME_OVER, INPUT_NAME
        self.snake = Snake()
        self.food = Food()
        self.ui = UI()
        
        self.score = 0
        self.player_name = ""
        self.new_highscore_rank = None
        self.highscores = load_highscores()
        
        self.current_difficulty_name = "MEDIUM"
        self.fps = DIFFICULTIES[self.current_difficulty_name]

    def handle_events(self):
        for event in pygame.event.get():
            if event.type == pygame.QUIT:
                pygame.quit()
                sys.exit()

            if self.state == "MENU":
                if event.type == pygame.KEYDOWN:
                    if event.key == pygame.K_SPACE:
                        self.state = "PLAYING"
                        self.snake.reset()
                        self.score = 0
                        self.food.respawn()
                    elif event.key == pygame.K_LEFT:
                        self._change_difficulty(-1)
                    elif event.key == pygame.K_RIGHT:
                        self._change_difficulty(1)

            elif self.state == "PLAYING":
                if event.type == pygame.KEYDOWN:
                    self.snake.handle_input(event.key)

            elif self.state == "GAME_OVER":
                if event.type == pygame.KEYDOWN:
                    if self.new_highscore_rank is None:
                        if event.key == pygame.K_r:
                            self.state = "MENU"
                        elif event.key == pygame.K_q:
                            pygame.quit()
                            sys.exit()
                    
                    if self.new_highscore_rank is not None:
                         # Gestion de la saisie du nom
                        if event.key == pygame.K_RETURN:
                            if self.player_name:
                                self.highscores = save_highscore(self.player_name, self.score, self.highscores)
                                self.new_highscore_rank = None
                                self.state = "MENU"
                        elif event.key == pygame.K_BACKSPACE:
                            self.player_name = self.player_name[:-1]
                        else:
                            # Limiter la longueur du nom et caractères valides
                            if len(self.player_name) < 10 and event.unicode.isalnum():
                                self.player_name += event.unicode

    def _change_difficulty(self, direction):
        difficulties = list(DIFFICULTIES.keys())
        current_index = difficulties.index(self.current_difficulty_name)
        new_index = (current_index + direction) % len(difficulties)
        self.current_difficulty_name = difficulties[new_index]
        self.fps = DIFFICULTIES[self.current_difficulty_name]

    def update(self):
        if self.state == "PLAYING":
            self.snake.move()
            
            # Manger la nourriture
            if self.snake.body[0] == self.food.position:
                self.snake.grow = True
                self.score += 1
                self.food.respawn()
                # Empêcher la nourriture d'apparaître sur le serpent
                while self.food.position in self.snake.body:
                    self.food.respawn()

            # Collisions
            if self.snake.check_collision():
                self.state = "GAME_OVER"
                self._check_highscore()

    def _check_highscore(self):
        # Vérifie si le score actuel mérite d'être dans le top 3
        if len(self.highscores) < 3 or self.score > self.highscores[-1]['score']:
            self.new_highscore_rank = True # Juste un flag dire qu'on a un highscore
            self.player_name = "" 
        else:
            self.new_highscore_rank = None

    def draw(self):
        self.screen.fill(BG_COLOR)

        if self.state == "MENU":
            self.ui.draw_menu(self.screen, self.current_difficulty_name)
        
        elif self.state == "PLAYING":
            self.snake.draw(self.screen)
            self.food.draw(self.screen)
            self.ui.draw_score(self.screen, self.score, self.highscores)
            
        elif self.state == "GAME_OVER":
            self.ui.draw_game_over(self.screen, self.score, self.new_highscore_rank, self.highscores)
            if self.new_highscore_rank is not None:
                self.ui.draw_input(self.screen, self.player_name)

        pygame.display.flip()

    def run(self):
        while True:
            self.handle_events()
            self.update()
            self.draw()
            self.clock.tick(self.fps)
