import pygame
import sys
from src.model import GameModel
from src.view import GameView
from src.generator import LevelGenerator
import random

class GameController:
    def __init__(self):
        self.model = GameModel()
        self.view = GameView(self.model)
        self.clock = pygame.time.Clock()
        self.generator = LevelGenerator(width=10, height=10) # 10x10 grid

    def run(self):
        # Initial State
        state = "MENU"

        current_level_idx = self.load_progress()
        

        
        # Pre-load first level to get dimensions for window init (or set default)
        # For now, let's load level 1 to init view
        try:
            self.model.load_level(f'levels/level{current_level_idx}.txt')
        except FileNotFoundError:
            # Fallback or error
            pass
            
        self.view.initialize()

        running = True
        while running:
            if state == "MENU":
                for event in pygame.event.get():
                    if event.type == pygame.QUIT:
                        running = False
                    elif event.type == pygame.KEYDOWN:
                        if event.key == pygame.K_RETURN:
                            state = "GAME"
                            self.model.load_level(f'levels/level{current_level_idx}.txt')
                            self.view.initialize() # Re-init if size changes
                        elif event.key == pygame.K_g:
                            state = "GAME"
                            self.load_random_level()
                        elif event.key == pygame.K_ESCAPE:
                            running = False
                self.view.draw_menu()

            elif state == "GAME":
                for event in pygame.event.get():
                    if event.type == pygame.QUIT:
                        running = False
                    elif event.type == pygame.KEYDOWN:
                        if event.key == pygame.K_ESCAPE:
                            state = "MENU"
                        elif event.key == pygame.K_u:
                            self.model.undo()
                        elif event.key == pygame.K_r: # Restart
                            # Check if it was a random level? For now just reload from file if file-based, 
                            # or re-generate if random? The model doesn't store 'source' currently.
                            # Simpler: Just reload the current level index file if standard, 
                            # or do nothing/warn if random.
                            # For better UX, we'd need to store level source.
                            # Let's just reload file for now.
                            try:
                                self.model.load_level(f'levels/level{current_level_idx}.txt')
                            except:
                                pass # Random level cannot be restarted easily yet without saving to file
                        else:
                            dx, dy = 0, 0
                            if event.key == pygame.K_UP: dy = -1
                            elif event.key == pygame.K_DOWN: dy = 1
                            elif event.key == pygame.K_LEFT: dx = -1
                            elif event.key == pygame.K_RIGHT: dx = 1
                            
                            if dx != 0 or dy != 0:
                                self.model.move_player(dx, dy)
                
                self.view.draw_game()
                
                if self.model.check_win():
                    state = "VICTORY"

            elif state == "VICTORY":
                for event in pygame.event.get():
                    if event.type == pygame.QUIT:
                        running = False
                    elif event.type == pygame.KEYDOWN:
                        if event.key == pygame.K_RETURN:
                            current_level_idx += 1
                            self.save_progress(current_level_idx)
                            try:
                                self.model.load_level(f'levels/level{current_level_idx}.txt')
                                state = "GAME"
                                self.view.initialize()
                            except FileNotFoundError:
                                print("No more levels! Back to menu.")
                                state = "MENU"
                                current_level_idx = 1 # Reset to 1
                        elif event.key == pygame.K_ESCAPE:
                            state = "MENU"
                
                self.view.draw_victory()

            self.clock.tick(60)

        pygame.quit()
        sys.exit()

    def load_random_level(self):
        # Generate level lines
        # Increase difficulty? Fixed size for now.
        lines = self.generator.generate_level(num_boxes=3)
        
        # Save to temp file to reuse load_level logic, or refactor load_level.
        # Refactoring load_level to accept lines list is cleaner, but let's stick to file interface for minimal change
        temp_file = 'levels/temp_random.txt'
        with open(temp_file, 'w') as f:
            f.writelines(lines)
            
        self.model.load_level(temp_file)
        self.model.current_level = 999 # Special ID for random
        self.view.initialize()

    def load_progress(self):
        try:
            with open("save.txt", "r") as f:
                return int(f.read().strip())
        except (FileNotFoundError, ValueError):
            return 1

    def save_progress(self, level_idx):
        with open("save.txt", "w") as f:
            f.write(str(level_idx))
