import random

class LevelGenerator:
    def __init__(self, width=10, height=10):
        self.width = width
        self.height = height

    def generate_level(self, num_boxes=3, diff_steps=100):
        # 1. Initialize Solid Grid
        self.width = max(self.width, 8) 
        self.height = max(self.height, 8)
        
        # Grid: '#' = Wall, ' ' = Floor
        grid = [['#' for _ in range(self.width)] for _ in range(self.height)]
        
        # 2. Carve Room (Random Walk)
        # Use a random walker to make irregular shapes
        start_x, start_y = self.width // 2, self.height // 2
        floor_positions = {(start_x, start_y)}
        
        walkers = [(start_x, start_y)]
        for _ in range(200): # Carve 200 tiles roughly
            wx, wy = random.choice(walkers)
            # Random direction
            dx, dy = random.choice([(0, 1), (0, -1), (1, 0), (-1, 0)])
            nx, ny = wx + dx, wy + dy
            
            if 1 < nx < self.width - 2 and 1 < ny < self.height - 2:
                grid[ny][nx] = ' '
                floor_positions.add((nx, ny))
                walkers.append((nx, ny))
                
        # 3. Place Targets
        targets = []
        possible_targets = list(floor_positions)
        random.shuffle(possible_targets)
        
        for i in range(min(num_boxes, len(possible_targets))):
             targets.append(possible_targets[i])
             
        # Boxes start on targets (Solved state)
        boxes = list(targets)
        
        # Player starts adjacent to a random box to facilitate pulling
        available_starts = []
        for bx, by in boxes:
            for dx, dy in [(0, 1), (0, -1), (1, 0), (-1, 0)]:
                nx, ny = bx + dx, by + dy
                if (nx, ny) in floor_positions and (nx, ny) not in boxes:
                    available_starts.append((nx, ny))
        
        if not available_starts:
            # Fallback (should be rare with enough floor)
            return self.generate_level(num_boxes, diff_steps)

        px, py = random.choice(available_starts)
        
        # 4. Reverse Play (Pull boxes)
        
        for _ in range(diff_steps):
            # Check for possible pulls
            # Pull: Player (px, py) moves to (nx, ny). Box at (bx, by) moves to (px, py).
            # Constraint: Box (bx, by) must be adjacent to Player (px, py).
            # Constraint: (nx, ny) must be empty (Backing up space).
            # Constraint: (bx, by) is the box we are pulling.
            # Relation: nx = px + (px - bx), ny = py + (py - by).
            
            # Find boxes adjacent to player
            adjacent_boxes = []
            for dx, dy in [(0, 1), (0, -1), (1, 0), (-1, 0)]:
                bx, by = px + dx, py + dy
                if (bx, by) in boxes:
                     # Check space behind player to pull into
                     pull_x, pull_y = px - dx, py - dy
                     if (pull_x, pull_y) in floor_positions and (pull_x, pull_y) not in boxes: 
                         # We allow pulling onto targets, but let's avoid if possible for complexity? 
                         # No, standard sokoban allows moving boxes over targets.
                         adjacent_boxes.append(((bx, by), (pull_x, pull_y)))
            
            if adjacent_boxes:
                # Choose one
                (bx, by), (nx, ny) = random.choice(adjacent_boxes)
                
                # Perform Pull
                boxes.remove((bx, by))
                boxes.append((px, py)) # Box moves to where player was
                
                px, py = nx, ny # Player moves backward
            else:
                # If no pull possible, move player randomly to find another box
                # Simple random walk on floor
                neighbors = []
                for dx, dy in [(0, 1), (0, -1), (1, 0), (-1, 0)]:
                    nx, ny = px + dx, py + dy
                    if (nx, ny) in floor_positions and (nx, ny) not in boxes:
                        neighbors.append((nx, ny))
                
                if neighbors:
                    px, py = random.choice(neighbors)
                else:
                    break # Trapped

        # 5. Render
        final_grid = [['#' for _ in range(self.width)] for _ in range(self.height)]
        
        for y in range(self.height):
             for x in range(self.width):
                 if grid[y][x] == '#': final_grid[y][x] = '#'
                 else: final_grid[y][x] = ' '
        
        for tx, ty in targets:
            final_grid[ty][tx] = '.'
            
        for bx, by in boxes:
            if (bx, by) in targets:
                final_grid[by][bx] = '*'
            else:
                final_grid[by][bx] = '$'
                
        if (px, py) in targets:
             final_grid[py][px] = '+'
        else:
             final_grid[py][px] = '@'

        # Trim grid (remove excess walls)
        min_x, max_x = self.width, 0
        min_y, max_y = self.height, 0
        
        for y in range(self.height):
            for x in range(self.width):
                if final_grid[y][x] != '#':
                    min_x = min(min_x, x)
                    max_x = max(max_x, x)
                    min_y = min(min_y, y)
                    max_y = max(max_y, y)
                    
        # Add padding
        min_x = max(0, min_x - 1)
        max_x = min(self.width - 1, max_x + 1)
        min_y = max(0, min_y - 1)
        max_y = min(self.height - 1, max_y + 1)
        
        trimmed_lines = []
        for y in range(min_y, max_y + 1):
            row = "".join(final_grid[y][min_x:max_x+1])
            trimmed_lines.append(row + "\n")

        return trimmed_lines
