import pygame

class GameView:
    def __init__(self, model, tile_size=64):
        self.model = model
        self.tile_size = tile_size
        self.screen = None
        self.font = None

    def initialize(self):
        screen_width = self.model.width * self.tile_size
        screen_height = self.model.height * self.tile_size + 50 # Add space for HUD
        self.screen = pygame.display.set_mode((screen_width, screen_height))
        pygame.display.set_caption("Sokoban")
        self.font = pygame.font.SysFont(None, 36)
        
        # Load Assets
        self.images = {}
        try:
            self.images['wall'] = pygame.transform.scale(pygame.image.load('assets/wall.png'), (self.tile_size, self.tile_size))
            self.images['floor'] = pygame.transform.scale(pygame.image.load('assets/floor.png'), (self.tile_size, self.tile_size))
            self.images['box'] = pygame.transform.scale(pygame.image.load('assets/box.png'), (self.tile_size, self.tile_size))
            self.images['box_done'] = pygame.transform.scale(pygame.image.load('assets/box_done.png'), (self.tile_size, self.tile_size))
            self.images['target'] = pygame.transform.scale(pygame.image.load('assets/target.png'), (self.tile_size, self.tile_size))
            self.images['player'] = pygame.transform.scale(pygame.image.load('assets/player.png'), (self.tile_size, self.tile_size))
        except Exception as e:
            print(f"Error loading assets: {e}")
            # Fallback to shapes if needed, but we expect assets to exist.

    def draw_menu(self):
        self.screen.fill((30, 30, 30))
        title = self.font.render("SOKOBAN", True, (255, 255, 255))
        start_text = self.font.render("ENTER: Start / Next Level", True, (200, 200, 200)) # Updated text
        gen_text = self.font.render("G: Generate Random Level", True, (200, 200, 200))
        quit_text = self.font.render("ESC: Quit", True, (200, 200, 200))

        title_rect = title.get_rect(center=(self.screen.get_width() // 2, self.screen.get_height() // 4))
        start_rect = start_text.get_rect(center=(self.screen.get_width() // 2, self.screen.get_height() // 2))
        gen_rect = gen_text.get_rect(center=(self.screen.get_width() // 2, self.screen.get_height() // 2 + 40))
        quit_rect = quit_text.get_rect(center=(self.screen.get_width() // 2, self.screen.get_height() // 2 + 80))

        self.screen.blit(title, title_rect)
        self.screen.blit(start_text, start_rect)
        self.screen.blit(gen_text, gen_rect)
        self.screen.blit(quit_text, quit_rect)
        pygame.display.flip()

    def draw_victory(self):
        # Use a semi-transparent overlay if possible, or just a new screen
        # For simple MVC, let's just draw over
        if not hasattr(self, 'overlay'):
             self.overlay = pygame.Surface(self.screen.get_size())
             self.overlay.set_alpha(128)
             self.overlay.fill((0, 0, 0))
        
        self.screen.blit(self.overlay, (0, 0))

        text = self.font.render("VICTORY!", True, (0, 255, 0))
        subtext = self.font.render("Press ENTER for Next Level", True, (255, 255, 255))
        
        text_rect = text.get_rect(center=(self.screen.get_width() // 2, self.screen.get_height() // 2 - 20))
        subtext_rect = subtext.get_rect(center=(self.screen.get_width() // 2, self.screen.get_height() // 2 + 20))
        
        self.screen.blit(text, text_rect)
        self.screen.blit(subtext, subtext_rect)
        pygame.display.flip()

    def draw_game(self):
        self.screen.fill((50, 50, 50)) # Background

        # Draw HUD
        moves_text = self.font.render(f"Level: {self.model.current_level}   Moves: {self.model.moves}", True, (255, 255, 255))
        self.screen.blit(moves_text, (10, 10))
        
        # Calculate offset to center the grid or place it below HUD
        offset_y = 50 

        # Draw Floor (Background for grid)
        # We can tile the floor or just draw it where walls aren't
        for y in range(self.model.height):
            for x in range(self.model.width):
                self.screen.blit(self.images['floor'], (x * self.tile_size, y * self.tile_size + offset_y))

        # Draw Targets
        for x, y in self.model.targets:
             self.screen.blit(self.images['target'], (x * self.tile_size, y * self.tile_size + offset_y))

        # Draw Walls
        for x, y in self.model.walls:
             self.screen.blit(self.images['wall'], (x * self.tile_size, y * self.tile_size + offset_y))

        # Draw Boxes
        for x, y in self.model.boxes:
            if (x, y) in self.model.targets:
                self.screen.blit(self.images['box_done'], (x * self.tile_size, y * self.tile_size + offset_y))
            else:
                self.screen.blit(self.images['box'], (x * self.tile_size, y * self.tile_size + offset_y))

        # Draw Player
        if self.model.player_pos:
            x, y = self.model.player_pos
            self.screen.blit(self.images['player'], (x * self.tile_size, y * self.tile_size + offset_y))
        
        pygame.display.flip()
