#!/usr/bin/env python3
"""
Script de configuration automatique du swap, zram et swappiness.
Optimisé pour Linux et Raspberry Pi 5.

Usage:
    sudo python3 config_swap.py [--swap-size 4] [--zram-percent 50] [--swappiness 20] [--dry-run] [--yes]
"""

import os
import sys
import argparse
import subprocess
import shutil
import logging
from typing import List, Optional

# Configuration du logging
logging.basicConfig(level=logging.INFO, format="%(message)s")
logger = logging.getLogger(__name__)

class SystemConfigurator:
    """Gère la configuration du système (Swap, Zram, Swappiness)."""

    def __init__(self, dry_run: bool = False, force: bool = False):
        self.dry_run = dry_run
        self.force = force
        self.swapfile_path = "/swapfile"
        self.zramswap_conf = "/etc/default/zramswap"
        self.fstab_path = "/etc/fstab"
        self.sysctl_conf = "/etc/sysctl.conf"
        self.sysctl_d_path = "/etc/sysctl.d/99-swappiness.conf"

    def run_command(self, cmd: str) -> None:
        """Exécute une commande shell avec gestion du mode dry-run."""
        if self.dry_run:
            logger.info(f"[DRY-RUN] Exécution : {cmd}")
            return

        logger.info(f"> {cmd}")
        try:
            result = subprocess.run(cmd, shell=True, check=True, capture_output=True, text=True)
        except subprocess.CalledProcessError as e:
            logger.error(f"Erreur lors de l'exécution de '{cmd}': {e.stderr.strip()}")
            # On ne lève pas d'exception fatale ici pour permettre la suite si possible, 
            # mais dans un script critique, on pourrait décider d'arrêter.
            # Pour l'instant, on log l'erreur.

    def backup_file(self, filepath: str) -> None:
        """Crée une sauvegarde d'un fichier système."""
        if not os.path.exists(filepath):
            return

        backup_path = f"{filepath}.bak"
        if self.dry_run:
            logger.info(f"[DRY-RUN] Sauvegarde de {filepath} vers {backup_path}")
        else:
            try:
                shutil.copy2(filepath, backup_path)
                logger.info(f"Sauvegarde créée : {backup_path}")
            except OSError as e:
                logger.error(f"Impossible de sauvegarder {filepath}: {e}")

    def get_ram_gb(self) -> float:
        """Retourne la quantité totale de RAM en Go."""
        try:
            with open("/proc/meminfo") as f:
                for line in f:
                    if line.startswith("MemTotal"):
                        # MemTotal est en kB
                        kb = int(line.split()[1])
                        return kb / (1024 * 1024)
        except FileNotFoundError:
            logger.error("Impossible de lire /proc/meminfo")
            return 0.0
        return 0.0

    def configure_swapfile(self, size_gb: float) -> None:
        """Configure le fichier de swap."""
        logger.info(f"\n--- Configuration du Swapfile ({size_gb} Go) ---")
        
        # Désactiver le swap actuel
        self.run_command(f"swapoff {self.swapfile_path} 2>/dev/null || true")

        if not self.dry_run and os.path.exists(self.swapfile_path):
            self.run_command(f"rm -f {self.swapfile_path}")
        elif self.dry_run:
            logger.info(f"[DRY-RUN] Suppression de l'ancien swapfile {self.swapfile_path}")

        # Création du nouveau swapfile
        self.run_command(f"fallocate -l {int(size_gb)}G {self.swapfile_path}")
        self.run_command(f"chmod 600 {self.swapfile_path}")
        self.run_command(f"mkswap {self.swapfile_path}")
        self.run_command(f"swapon {self.swapfile_path}")

        # Mise à jour de fstab
        self.update_fstab()

    def update_fstab(self) -> None:
        """Met à jour /etc/fstab pour rendre le swap persistent."""
        if self.dry_run:
            logger.info(f"[DRY-RUN] Ajout du swapfile dans {self.fstab_path}")
            return

        self.backup_file(self.fstab_path)
        
        try:
            with open(self.fstab_path, "r") as f:
                content = f.read()
            
            if self.swapfile_path not in content:
                with open(self.fstab_path, "a") as f:
                    f.write(f"\n{self.swapfile_path} none swap sw 0 0\n")
                logger.info(f"Ajouté à {self.fstab_path}")
            else:
                logger.info(f"Déjà présent dans {self.fstab_path}")
        except OSError as e:
            logger.error(f"Erreur lors de la modification de {self.fstab_path}: {e}")

    def configure_zram(self, percent: int) -> None:
        """Installe et configure zram-tools."""
        logger.info(f"\n--- Configuration de zram ({percent}%) ---")
        
        self.run_command("apt update")
        self.run_command("apt purge -y rpi-swap") # Conflit potentiel
        self.run_command("apt install -y zram-tools")

        if self.dry_run:
            logger.info(f"[DRY-RUN] Configuration de {self.zramswap_conf} avec PERCENT={percent}")
            logger.info("[DRY-RUN] Redémarrage du service zramswap")
            return

        # Configuration du fichier
        try:
            lines = []
            if os.path.exists(self.zramswap_conf):
                with open(self.zramswap_conf, "r") as f:
                    lines = f.readlines()
            
            new_lines = []
            found = False
            for line in lines:
                if line.strip().startswith("PERCENT="):
                    new_lines.append(f"PERCENT={percent}\n")
                    found = True
                else:
                    new_lines.append(line)
            
            if not found:
                new_lines.append(f"PERCENT={percent}\n")
            
            with open(self.zramswap_conf, "w") as f:
                f.writelines(new_lines)
            logger.info(f"Fichier {self.zramswap_conf} mis à jour.")

        except OSError as e:
            logger.error(f"Erreur écriture {self.zramswap_conf}: {e}")
            return

        # Redémarrage du service
        self.run_command("systemctl stop zramswap.service")
        self.run_command("swapoff /dev/zram0 2>/dev/null || true")
        self.run_command("systemctl enable zramswap.service")
        self.run_command("systemctl start zramswap.service")

    def configure_swappiness(self, value: int) -> None:
        """Configure la swappiness."""
        logger.info(f"\n--- Configuration de Swappiness ({value}) ---")
        self.run_command(f"sysctl vm.swappiness={value}")

        if self.dry_run:
            logger.info(f"[DRY-RUN] Écriture de vm.swappiness={value} dans {self.sysctl_d_path} ou {self.sysctl_conf}")
            return

        # Priorité à /etc/sysctl.d/
        target_file = self.sysctl_d_path if os.path.isdir("/etc/sysctl.d") else self.sysctl_conf
        
        self.backup_file(target_file)

        try:
            if target_file == self.sysctl_d_path:
                with open(target_file, "w") as f:
                    f.write(f"vm.swappiness={value}\n")
            else:
                # Fallback sur sysctl.conf
                with open(target_file, "r") as f:
                    lines = f.readlines()
                
                # Supprimer les anciennes configs swappiness pour éviter les doublons
                lines = [l for l in lines if not l.strip().startswith("vm.swappiness")]
                lines.append(f"vm.swappiness={value}\n")
                
                with open(target_file, "w") as f:
                    f.writelines(lines)
            
            logger.info(f"Configuration sauvegardée dans {target_file}")
            
        except OSError as e:
            logger.error(f"Erreur lors de l'écriture de la configuration swappiness: {e}")

    def check_dependencies(self) -> None:
        """Vérifie que les outils nécessaires sont présents."""
        required_tools = ["fallocate", "mkswap", "swapon", "swapoff", "sysctl", "systemctl", "apt"]
        missing = []
        for tool in required_tools:
            if shutil.which(tool) is None:
                missing.append(tool)
        
        if missing:
            logger.error(f"Erreur : Outils manquants : {', '.join(missing)}")
            if not self.dry_run:
                sys.exit(1)
            else:
                logger.warning("[DRY-RUN] Outils manquants ignorés.")

    def restore(self) -> None:
        """Restaure les configurations depuis les backups (.bak)."""
        logger.info("\n--- Restauration de la configuration ---")
        
        # Restauration fstab
        if os.path.exists(f"{self.fstab_path}.bak"):
            if self.dry_run:
                logger.info(f"[DRY-RUN] Restauration de {self.fstab_path} depuis {self.fstab_path}.bak")
            else:
                shutil.copy2(f"{self.fstab_path}.bak", self.fstab_path)
                logger.info(f"Restauré : {self.fstab_path}")
        else:
            logger.warning(f"Aucun backup trouvé pour {self.fstab_path}")

        # Restauration sysctl
        # On vérifie les deux emplacements possibles
        files_to_check = [self.sysctl_d_path, self.sysctl_conf]
        for fpath in files_to_check:
             if os.path.exists(f"{fpath}.bak"):
                if self.dry_run:
                    logger.info(f"[DRY-RUN] Restauration de {fpath} depuis {fpath}.bak")
                else:
                    shutil.copy2(f"{fpath}.bak", fpath)
                    logger.info(f"Restauré : {fpath}")

        # Désactivation du swap
        self.run_command(f"swapoff {self.swapfile_path} 2>/dev/null || true")
        logger.info("Restauration terminée. Redémarrez pour appliquer complètement les changements noyaux (zram, etc).")

    def uninstall(self) -> None:
        """Désinstalle complètement la configuration (Swap, Zram, Swappiness)."""
        logger.info("\n--- Désinstallation complète ---")

        # 1. Swapfile
        self.run_command(f"swapoff {self.swapfile_path} 2>/dev/null || true")
        if self.dry_run:
             logger.info(f"[DRY-RUN] Suppression de {self.swapfile_path}")
             logger.info(f"[DRY-RUN] Nettoyage de {self.fstab_path}")
        else:
            if os.path.exists(self.swapfile_path):
                os.remove(self.swapfile_path)
                logger.info(f"Supprimé : {self.swapfile_path}")
            
            # Nettoyage fstab
            try:
                with open(self.fstab_path, "r") as f:
                    lines = f.readlines()
                with open(self.fstab_path, "w") as f:
                    for line in lines:
                        if self.swapfile_path not in line:
                            f.write(line)
                logger.info(f"Nettoyé : {self.fstab_path}")
            except OSError as e:
                logger.error(f"Erreur nettoyage fstab: {e}")

        # 2. Zram
        self.run_command("systemctl stop zramswap.service")
        self.run_command("systemctl disable zramswap.service 2>/dev/null || true")
        self.run_command("apt purge -y zram-tools")
        
        # 3. Swappiness (Retour défaut 60)
        self.run_command("sysctl vm.swappiness=60")
        if self.dry_run:
             logger.info(f"[DRY-RUN] Suppression config swappiness dans {self.sysctl_d_path}")
        else:
            if os.path.exists(self.sysctl_d_path):
                os.remove(self.sysctl_d_path)
                logger.info(f"Supprimé : {self.sysctl_d_path}")
            # Note: On ne nettoie pas sysctl.conf automatiquement pour éviter de casser d'autres configs, 
            # sauf si on a fait un restore avant.

        logger.info("Désinstallation terminée.")

    def verify(self) -> None:
        """Affiche l'état final."""
        if self.dry_run:
            logger.info("\n[DRY-RUN] Vérification simulée.")
            return

        print("\n✅ État actuel :")
        subprocess.run("swapon --show", shell=True)
        print("\nSwappiness : ", end="")
        subprocess.run("cat /proc/sys/vm/swappiness", shell=True)
        if os.path.exists("/sys/block/zram0/disksize"):
            print("Zram info : ", end="")
            subprocess.run("cat /sys/block/zram0/disksize", shell=True)

def parse_args():
    parser = argparse.ArgumentParser(description="Configuration optimisée du Swap et Zram.")
    parser.add_argument("--swap-size", type=float, help="Taille du Swapfile en Go")
    parser.add_argument("--zram-percent", type=int, help="Pourcentage de RAM pour Zram (ex: 50)")
    parser.add_argument("--swappiness", type=int, help="Valeur de swappiness (0-100)")
    parser.add_argument("--dry-run", action="store_true", help="Simulation (ne modifie rien)")
    parser.add_argument("--restore", action="store_true", help="Restaurer la configuration précédente (fstab, sysctl)")
    parser.add_argument("--uninstall", action="store_true", help="Tout désinstaller (Swap, Zram, Swappiness)")
    parser.add_argument("--yes", "-y", action="store_true", help="Accepter automatiquement les valeurs proposées")
    return parser.parse_args()


def main():
    args = parse_args()

    # En mode dry-run, on peut être non-root, mais certaines lectures peuvent échouer.
    if os.geteuid() != 0 and not args.dry_run:
        print("Erreur : Ce script doit être exécuté en root (sudo).")
        sys.exit(1)

    configurator = SystemConfigurator(dry_run=args.dry_run)
    configurator.check_dependencies()

    if args.restore:
        if not args.yes and not args.dry_run:
             if input("Voulez-vous vraiment restaurer les fichiers de configuration ? [y/N] ").lower() != 'y':
                 print("Annulé.")
                 sys.exit(0)
        configurator.restore()
        sys.exit(0)

    if args.uninstall:
        if not args.yes and not args.dry_run:
             if input("Voulez-vous vraiment TOUT désinstaller (Swap, Zram, Configs) ? [y/N] ").lower() != 'y':
                 print("Annulé.")
                 sys.exit(0)
        configurator.uninstall()
        sys.exit(0)
    
    ram_gb = configurator.get_ram_gb()
    print(f"RAM totale détectée : {ram_gb:.2f} Go\n")

    # Valeurs par défaut intelligentes
    default_swap_size = int(ram_gb) if ram_gb > 1 else 1
    default_zram_percent = 50
    default_swappiness = 20

    # Détermination des valeurs finales
    swap_size = args.swap_size if args.swap_size is not None else default_swap_size
    zram_percent = args.zram_percent if args.zram_percent is not None else default_zram_percent
    swappiness = args.swappiness if args.swappiness is not None else default_swappiness

    # Mode interactif si pas d'arguments forcés ni --yes
    if not args.yes and not any([args.swap_size, args.zram_percent, args.swappiness, args.dry_run]):
        print("💡 Valeurs proposées :")
        print(f"- Swapfile : {swap_size} Go")
        print(f"- Zram : {zram_percent}%")
        print(f"- Swappiness : {swappiness}\n")
        
        try:
            in_swap = input(f"Taille swapfile [Go] (Enter = {swap_size}) : ")
            if in_swap: swap_size = float(in_swap)
            
            in_zram = input(f"Zram % (Enter = {zram_percent}) : ")
            if in_zram: zram_percent = int(in_zram)
            
            in_swapp = input(f"Swappiness (Enter = {swappiness}) : ")
            if in_swapp: swappiness = int(in_swapp)
        except ValueError:
            print("Entrée invalide.")
            sys.exit(1)

    print(f"\n🚀 Application de la configuration...")
    if args.dry_run:
        print("🔧 MODE DRY-RUN : Aucune modification ne sera appliquée.\n")

    configurator.configure_swapfile(swap_size)
    configurator.configure_zram(zram_percent)
    configurator.configure_swappiness(swappiness)
    
    configurator.verify()
    print("\n✨ Terminé.")

if __name__ == "__main__":
    main()
