#!/usr/bin/env python3
import tkinter as tk
from tkinter import ttk, messagebox, scrolledtext
import logging
import threading
import sys
import os
import json

# Import du backend
try:
    from config_swap import SystemConfigurator
except ImportError:
    # Fallback pour le développement ou si lancé depuis ailleurs
    sys.path.append(os.path.dirname(os.path.abspath(__file__)))
    from config_swap import SystemConfigurator

# --- Configuration du Thème Sombre ---
COLOR_BG = "#2b2b2b"
COLOR_FG = "#ffffff"
COLOR_ACCENT = "#007acc"
COLOR_SECONDARY = "#3c3f41"
COLOR_TEXT_DIM = "#aaaaaa"
FONT_TITLE = ("Helvetica", 14, "bold")
FONT_NORMAL = ("Helvetica", 10)
FONT_SMALL = ("Helvetica", 9, "italic")

CONFIG_FILE = os.path.join(os.path.dirname(os.path.abspath(__file__)), "window_config.json")

class TextHandler(logging.Handler):
    """Redirige les logs vers un widget Text Tkinter."""
    def __init__(self, text_widget):
        super().__init__()
        self.text_widget = text_widget

    def emit(self, record):
        msg = self.format(record)
        def append():
            self.text_widget.configure(state='normal')
            self.text_widget.insert(tk.END, msg + '\n')
            self.text_widget.see(tk.END)
            self.text_widget.configure(state='disabled')
        self.text_widget.after(0, append)

class HelpWindow(tk.Toplevel):
    def __init__(self, parent):
        super().__init__(parent)
        self.title("Guide du Débutant - Comprendre la Mémoire")
        self.geometry("700x600")
        self.configure(bg=COLOR_BG)
        
        # Titre Principal
        tk.Label(self, text="🎓 Comprendre la mémoire sous Linux", bg=COLOR_BG, fg=COLOR_ACCENT, font=("Helvetica", 16, "bold")).pack(pady=10)

        # Zone de texte défilante
        text_area = scrolledtext.ScrolledText(self, bg=COLOR_BG, fg=COLOR_FG, font=("Helvetica", 11), wrap="word", padx=15, pady=15, borderwidth=0)
        text_area.pack(fill="both", expand=True)

        # Configuration des tags pour le formatage
        text_area.tag_config("h1", font=("Helvetica", 13, "bold"), foreground=COLOR_ACCENT, spacing1=15, spacing3=5)
        text_area.tag_config("h2", font=("Helvetica", 11, "bold"), foreground="white", spacing1=10)
        text_area.tag_config("b", font=("Helvetica", 11, "bold"))
        text_area.tag_config("i", font=("Helvetica", 11, "italic"))

        # Contenu Pédagogique
        content = [
            ("h1", "1. La RAM : Votre Plan de Travail"),
            ("text", "Imaginez que votre ordinateur est un bureau. La "),
            ("b", "RAM"),
            ("text", " est la surface de votre plan de travail. C'est là que vous posez les dossiers (logiciels) sur lesquels vous travaillez actuellement.\n\nPlus le bureau est grand, plus vous pouvez ouvrir de dossiers en même temps sans devoir les ranger."),

            ("h1", "2. Le Swapfile : L'Armoire à Archives"),
            ("text", "Le "),
            ("b", "Swapfile"),
            ("text", " (ou fichier d'échange) est comme une armoire à côté de votre bureau. Quand votre plan de travail (RAM) est plein, l'ordinateur doit ranger temporairement certains dossiers dans l'armoire pour faire de la place.\n\n"),
            ("i", "Problème :"),
            ("text", " Se lever pour aller chercher un dossier dans l'armoire est beaucoup plus lent que de l'avoir sous la main. C'est pour ça que le Swap ralentit l'ordinateur s'il est trop utilisé."),

            ("h1", "3. Zram : Le Sac de Compression"),
            ("text", "C'est une astuce géniale ! Au lieu de ranger les dossiers dans l'armoire lente, "),
            ("b", "Zram"),
            ("text", " prend le dossier, le compresse (comme un sac sous vide) et le laisse dans un petit coin du bureau (RAM).\n\n"),
            ("b", "Avantage :"),
            ("text", " Décompresser le dossier est ultra-rapide, beaucoup plus rapide que d'aller à l'armoire. Cela booste les performances, surtout sur les Raspberry Pi ou les vieux PC."),

            ("h1", "4. Swappiness : L'Impatience du Système"),
            ("text", "C'est un réglage (de 0 à 100) qui détermine l'empressement de votre ordinateur à utiliser l'armoire (Swap).\n\n"),
            ("text", "• "), ("b", "Faible (0-10) :"), ("text", " Le système est 'fainéant'. Il attend que le bureau soit VRAIMENT plein à craquer avant de ranger quelque chose. C'est bon pour la réactivité.\n"),
            ("text", "• "), ("b", "Élevé (60-100) :"), ("text", " Le système est 'maniaque'. Dès qu'un dossier n'est pas utilisé depuis 5 minutes, il le range dans l'armoire, même s'il y a encore de la place sur le bureau."),
            
            ("h1", "5. Recommandations pour ce projet"),
            ("text", "• "), ("b", "Swapfile :"), ("text", " Gardez une taille raisonnable (égale à votre RAM). C'est votre sécurité anti-crash.\n"),
            ("text", "• "), ("b", "Zram :"), ("text", " Activez-le toujours (50% ou plus). C'est de la performance gratuite !\n"),
            ("text", "• "), ("b", "Swappiness :"), ("text", " Réglez sur 10-20. On veut utiliser la RAM rapide au maximum avant de toucher au disque."),
        ]

        # Insertion du texte formaté
        text_area.configure(state='normal')
        for i, item in enumerate(content):
            tag, text = item
            
            # Ajout d'un saut de ligne avant les titres (sauf le premier)
            if tag == "h1" and i > 0:
                text_area.insert(tk.END, "\n")

            if tag == "text":
                text_area.insert(tk.END, text)
            else:
                text_area.insert(tk.END, text, tag)
                if tag == "h1":
                    text_area.insert(tk.END, "\n")
        
        text_area.configure(state='disabled')
        
        # Bouton fermer
        tk.Button(self, text="J'ai compris !", bg=COLOR_ACCENT, fg="white", font=("Helvetica", 12, "bold"), command=self.destroy).pack(pady=10)


class SwapGUI(tk.Tk):
    def __init__(self):
        super().__init__()
        self.title("Optimiseur de Mémoire (Swap/Zram)")
        self.configure(bg=COLOR_BG)
        self.load_window_config()
        
        # Vérification root
        self.is_root = os.geteuid() == 0
        if not self.is_root:
            messagebox.showwarning("Attention", "Ce programme doit être lancé avec 'sudo' pour appliquer les changements.\nMode lecture seule actif.")

        self.configurator = SystemConfigurator(dry_run=not self.is_root)
        self.ram_gb = self.configurator.get_ram_gb()
        
        self.create_widgets()
        self.setup_logging()
        
        self.protocol("WM_DELETE_WINDOW", self.on_closing)

    def load_window_config(self):
        """Charge la position et la taille de la fenêtre."""
        default_geometry = "600x750"
        try:
            if os.path.exists(CONFIG_FILE):
                with open(CONFIG_FILE, "r") as f:
                    config = json.load(f)
                    geometry = config.get("geometry", default_geometry)
                    self.geometry(geometry)
            else:
                self.geometry(default_geometry)
        except Exception as e:
            print(f"Erreur chargement config fenêtre: {e}")
            self.geometry(default_geometry)

    def save_window_config(self):
        """Sauvegarde la position et la taille de la fenêtre."""
        try:
            config = {"geometry": self.geometry()}
            with open(CONFIG_FILE, "w") as f:
                json.dump(config, f)
        except Exception as e:
            print(f"Erreur sauvegarde config fenêtre: {e}")

    def on_closing(self):
        self.save_window_config()
        self.destroy()

    def show_help(self):
        """Ouvre la fenêtre d'aide."""
        HelpWindow(self)

    def create_widgets(self):
        # Style
        style = ttk.Style()
        style.theme_use('clam')
        style.configure("TFrame", background=COLOR_BG)
        style.configure("TLabel", background=COLOR_BG, foreground=COLOR_FG, font=FONT_NORMAL)
        style.configure("TButton", background=COLOR_SECONDARY, foreground=COLOR_FG, borderwidth=1)
        style.map("TButton", background=[('active', COLOR_ACCENT)])
        style.configure("Horizontal.TScale", background=COLOR_BG, troughcolor=COLOR_SECONDARY, sliderlength=20)

        # Header
        header_frame = ttk.Frame(self)
        header_frame.pack(fill="x", padx=20, pady=10)
        
        # Titre et Bouton Aide dans le header
        title_frame = ttk.Frame(header_frame)
        title_frame.pack(side="left")
        
        lbl_title = tk.Label(title_frame, text="Optimiseur de Mémoire", bg=COLOR_BG, fg=COLOR_FG, font=FONT_TITLE)
        lbl_title.pack(side="left")
        
        btn_help = tk.Button(title_frame, text="?", bg=COLOR_ACCENT, fg="white", font=("Helvetica", 10, "bold"), width=3, command=self.show_help)
        btn_help.pack(side="left", padx=10)
        
        lbl_ram = tk.Label(header_frame, text=f"RAM : {self.ram_gb:.2f} Go", bg=COLOR_BG, fg=COLOR_ACCENT, font=FONT_TITLE)
        lbl_ram.pack(side="right")

        # --- Section 1: Swapfile ---
        self.create_section(
            "1. Mémoire Virtuelle (Swapfile)",
            "Le Swap est une extension de la RAM sur le disque dur/SD. Utile pour éviter les crashs, mais plus lent que la RAM.\nRecommandé : ~1x la taille de la RAM.",
            0, int(self.ram_gb * 2) + 1, self.ram_gb, "Go",
            "swap_var"
        )

        # --- Section 2: Zram ---
        self.create_section(
            "2. Compression RAM (Zram)",
            "Zram compresse une partie des données en RAM pour gagner de la place sans ralentir le système (contrairement au swap disque).\nRecommandé : 50% ou plus.",
            0, 100, 50, "%",
            "zram_var"
        )

        # --- Section 3: Swappiness ---
        self.create_section(
            "3. Priorité (Swappiness)",
            "Définit à quel moment le système commence à utiliser le swap.\nFaible (0-10) = Garder tout en RAM (Performance).\nÉlevé (60-100) = Libérer la RAM proactivement.",
            0, 100, 20, "",
            "swappiness_var"
        )

        # --- Log Area ---
        log_frame = ttk.Frame(self)
        log_frame.pack(fill="both", expand=True, padx=20, pady=10)
        tk.Label(log_frame, text="Journal d'opérations :", bg=COLOR_BG, fg=COLOR_TEXT_DIM, font=FONT_SMALL).pack(anchor="w")
        
        self.log_text = scrolledtext.ScrolledText(log_frame, height=8, bg=COLOR_SECONDARY, fg=COLOR_FG, font=("Consolas", 9))
        self.log_text.pack(fill="both", expand=True)
        self.log_text.configure(state='disabled')

        # --- Buttons ---
        btn_frame = ttk.Frame(self)
        btn_frame.pack(fill="x", padx=20, pady=20)

        self.btn_apply = tk.Button(btn_frame, text="APPLIQUER LA CONFIGURATION", bg=COLOR_ACCENT, fg="white", font=("Helvetica", 10, "bold"), command=self.apply_config)
        self.btn_apply.pack(side="right", padx=5)

        self.btn_restore = tk.Button(btn_frame, text="Restaurer Défaut", bg=COLOR_SECONDARY, fg=COLOR_FG, command=self.restore_config)
        self.btn_restore.pack(side="left", padx=5)
        
        self.btn_uninstall = tk.Button(btn_frame, text="Désinstaller Tout", bg="#d9534f", fg="white", command=self.uninstall_config)
        self.btn_uninstall.pack(side="left", padx=5)

    def create_section(self, title, description, min_val, max_val, default_val, unit, var_attr_name):
        frame = ttk.Frame(self)
        frame.pack(fill="x", padx=20, pady=10)

        # Titre
        tk.Label(frame, text=title, bg=COLOR_BG, fg=COLOR_FG, font=("Helvetica", 11, "bold")).pack(anchor="w")
        
        # Description
        tk.Label(frame, text=description, bg=COLOR_BG, fg=COLOR_TEXT_DIM, font=FONT_SMALL, justify="left").pack(anchor="w", pady=(0, 5))

        # Slider Container
        slider_frame = ttk.Frame(frame)
        slider_frame.pack(fill="x")

        # Variable
        var = tk.DoubleVar(value=default_val)
        setattr(self, var_attr_name, var) # Sauvegarde la variable dans l'instance

        # Label Valeur
        val_label = tk.Label(slider_frame, text=f"{int(default_val)} {unit}", bg=COLOR_BG, fg=COLOR_ACCENT, font=("Helvetica", 10, "bold"), width=8)
        val_label.pack(side="right")

        # Scale
        # Utilisation de default=val_label pour capturer la variable dans la lambda
        def update_label(v, label=val_label, unit=unit):
            label.config(text=f"{int(float(v))} {unit}")

        scale = ttk.Scale(slider_frame, from_=min_val, to=max_val, variable=var, orient="horizontal", command=update_label)
        scale.pack(side="left", fill="x", expand=True, padx=(0, 10))

    def setup_logging(self):
        handler = TextHandler(self.log_text)
        formatter = logging.Formatter('%(message)s')
        handler.setFormatter(formatter)
        
        # On récupère le logger de config_swap
        logger = logging.getLogger("config_swap")
        logger.addHandler(handler)
        logger.setLevel(logging.INFO)

    def threaded_task(self, task_func):
        """Lance une tâche en arrière-plan pour ne pas geler l'interface."""
        if not self.is_root:
             messagebox.showerror("Erreur", "Vous devez être ROOT (sudo) pour effectuer cette action.\nLancez 'sudo python3 swap_gui.py'")
             return

        self.btn_apply.config(state="disabled")
        self.btn_restore.config(state="disabled")
        self.btn_uninstall.config(state="disabled")
        
        def run():
            try:
                task_func()
                messagebox.showinfo("Succès", "Opération terminée avec succès !")
            except Exception as e:
                logging.error(f"Erreur critique : {e}")
                messagebox.showerror("Erreur", f"Une erreur est survenue : {e}")
            finally:
                self.after(0, lambda: self.reset_buttons())

        threading.Thread(target=run, daemon=True).start()

    def reset_buttons(self):
        self.btn_apply.config(state="normal")
        self.btn_restore.config(state="normal")
        self.btn_uninstall.config(state="normal")

    def apply_config(self):
        swap = self.swap_var.get()
        zram = int(self.zram_var.get())
        swappiness = int(self.swappiness_var.get())
        
        def task():
            logging.info("--- DÉBUT DE LA CONFIGURATION ---")
            self.configurator.configure_swapfile(swap)
            self.configurator.configure_zram(zram)
            self.configurator.configure_swappiness(swappiness)
            self.configurator.verify()
            logging.info("--- FIN ---")
            
        self.threaded_task(task)

    def restore_config(self):
        if messagebox.askyesno("Confirmer", "Voulez-vous restaurer la configuration précédente ?"):
            self.threaded_task(self.configurator.restore)

    def uninstall_config(self):
        if messagebox.askyesno("Danger", "Voulez-vous vraiment TOUT désinstaller (Swap, Zram, Configs) ?"):
            self.threaded_task(self.configurator.uninstall)

if __name__ == "__main__":
    app = SwapGUI()
    app.mainloop()
